"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const client_1 = require("@prisma/client");
const supabase_service_1 = require("../src/auth/supabase.service");
const data_1 = require("./mock/data");
const crypto_1 = require("crypto");
const prisma = new client_1.PrismaClient();
const supabaseService = new supabase_service_1.default();
async function seed() {
    try {
        let resetDb = false;
        let updateReferenceIds = true;
        if (resetDb) {
            await resetDatabase();
        }
        if (updateReferenceIds) {
            await updateExistingOrders();
        }
        console.log('Seed data created successfully');
    }
    catch (error) {
        console.error('Error seeding data:', error);
    }
    finally {
        await prisma.$disconnect();
    }
}
async function resetDatabase() {
    await prisma.orderDetailOptions.deleteMany();
    await prisma.orderDetails.deleteMany();
    await prisma.transaction.deleteMany();
    await prisma.order.deleteMany();
    await prisma.option.deleteMany();
    await prisma.menuOption.deleteMany();
    await prisma.menu.deleteMany();
    await prisma.merchant.deleteMany();
    await prisma.user.deleteMany();
    await prisma.role.deleteMany();
    await prisma.tier.deleteMany();
    for (const role of data_1.sampleData.roles) {
        await prisma.role.create({
            data: {
                id: role.id,
                name: role.name
            }
        });
    }
    for (const data of data_1.sampleData.users) {
        await supabaseService.deleteUserFromAuth(data.email);
        await supabaseService.signUpWithEmail(data.email, data.password);
        await prisma.user.create({
            data: {
                id: data.id,
                email: data.email,
                name: data.name,
                phoneNumber: data.phoneNumber,
                dateOfBirth: new Date(data.dateOfBirth),
                role: {
                    connect: {
                        id: data.roleId
                    }
                },
                address: {
                    ...(data.address)
                }
            }
        });
    }
    for (const data of data_1.sampleData.merchants) {
        await supabaseService.deleteUserFromAuth(data.email);
        await supabaseService.signUpWithEmail(data.email, data.password);
        await prisma.user.create({
            data: {
                id: data.id,
                email: data.email,
                name: data.name,
                phoneNumber: data.phoneNumber,
                dateOfBirth: new Date(data.dateOfBirth),
                role: {
                    connect: {
                        id: data.roleId
                    }
                },
                merchant: {
                    ...(data.merchant)
                }
            }
        });
    }
    var menuid = "";
    for (const menuData of data_1.sampleData.menus) {
        const createdMenu = await prisma.menu.create({
            data: {
                name: menuData.name,
                basePrice: menuData.basePrice,
                category: menuData.category,
                description: menuData.description,
                merchant: { connect: { id: menuData.merchantId } },
                image: menuData.image
            },
        });
        menuid = createdMenu.id;
        for (const menuOptionData of menuData.menuOptions) {
            const createdMenuOption = await prisma.menuOption.create({
                data: {
                    name: menuOptionData.name,
                    isRequired: menuOptionData.isRequired,
                    min: menuOptionData.min,
                    max: menuOptionData.max,
                    menu: { connect: { id: createdMenu.id } },
                },
            });
            for (const optionData of menuOptionData.options) {
                await prisma.option.create({
                    data: {
                        name: optionData.name,
                        price: optionData.price,
                        menuOption: { connect: { id: createdMenuOption.id } },
                    },
                });
            }
        }
    }
    for (const orderData of data_1.sampleData.orders) {
        const order = await prisma.order.create({
            data: {
                ...orderData,
                referenceId: generateReferenceId(),
                orderDetails: {
                    create: orderData.orderDetails.map((orderDetail) => ({
                        quantity: orderDetail.quantity,
                        basePrice: orderDetail.basePrice,
                        orderDetailOptions: {
                            create: orderDetail.orderDetailOptions.map((orderDetailOption) => ({
                                type: orderDetailOption.type,
                                name: orderDetailOption.name,
                                price: orderDetailOption.price,
                            })),
                        },
                        menu: { connect: { id: menuid } },
                    })),
                },
            },
        });
        for (const transactionsData of data_1.sampleData.transactions) {
            await prisma.transaction.create({
                data: {
                    ...transactionsData,
                    orderId: order.id,
                },
            });
        }
    }
    for (const creditsData of data_1.sampleData.credit) {
        const credit = await prisma.credit.findUnique({
            where: {
                id: creditsData.id
            }
        });
        if (!credit) {
            await prisma.credit.create({
                data: {
                    ...creditsData,
                },
            });
        }
    }
    for (const creditTransactionsData of data_1.sampleData.creditHistory) {
        const creditTransaction = await prisma.creditHistory.findUnique({
            where: {
                id: creditTransactionsData.id
            }
        });
        if (!creditTransaction) {
            await prisma.creditHistory.create({
                data: {
                    ...creditTransactionsData,
                },
            });
        }
    }
    await prisma.tier.createMany({
        data: [
            { name: 'Silver', pointsRequired: 0, pointsMultiplier: 1.0 },
            { name: 'Gold', pointsRequired: 600, pointsMultiplier: 1.5 },
            { name: 'Platinum', pointsRequired: 1000, pointsMultiplier: 2.0 },
        ],
    });
    const silverTier = await prisma.tier.findUnique({
        where: { name: 'Silver' },
    });
    if (!silverTier) {
        throw new Error('Silver tier not found');
    }
    await prisma.user.updateMany({
        data: {
            membershipTierId: silverTier.id,
        },
    });
    console.log('All users have been updated to Silver tier');
}
function generateReferenceId() {
    const now = new Date();
    const year = now.getFullYear().toString().slice(-2);
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const datePrefix = `${year}${month}${day}`;
    const randomSuffix = (0, crypto_1.randomBytes)(4).toString('hex');
    return `ORD-${datePrefix}-${randomSuffix}`;
}
async function updateExistingOrders() {
    const ordersWithoutRefId = await prisma.order.findMany({
        where: { referenceId: null },
    });
    for (const order of ordersWithoutRefId) {
        let referenceId;
        let exists;
        do {
            referenceId = generateReferenceId();
            const res = await prisma.order.findUnique({
                where: { referenceId },
            });
            exists = !!res;
        } while (exists);
        await prisma.order.update({
            where: { id: order.id },
            data: { referenceId },
        });
        console.log(`Updated Order ID ${order.id} with Reference ID ${referenceId}`);
    }
}
seed();
//# sourceMappingURL=seed.js.map