"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrderService = void 0;
const common_1 = require("@nestjs/common");
const prisma_1 = require("../prisma");
const membership_service_1 = require("../membership/membership.service");
const membership_dto_1 = require("../membership/membership.dto");
const utils_1 = require("../utils");
let OrderService = class OrderService {
    constructor(membershipService) {
        this.membershipService = membershipService;
    }
    async create(data) {
        let totalPrice = 0;
        const orderDetails = await Promise.all(data.orderDetails.map(async (orderDetail) => {
            const menu = await prisma_1.default.menu.findUnique({ where: { id: orderDetail.menuId } });
            const basePrice = menu.basePrice;
            const optionPrices = await Promise.all(orderDetail.orderDetailOptions.map(async (detailOption) => {
                const option = await prisma_1.default.option.findUnique({ where: { id: detailOption.optionId } });
                return option.price;
            }));
            const totalOptionPrice = optionPrices.reduce((acc, price) => acc + price, 0);
            const totalDetailPrice = (basePrice + totalOptionPrice) * orderDetail.quantity;
            totalPrice += totalDetailPrice;
            return {
                basePrice,
                quantity: orderDetail.quantity,
                menuId: orderDetail.menuId,
                orderDetailOptions: {
                    create: await Promise.all(orderDetail.orderDetailOptions.map(async (detailOption) => {
                        const menuOption = await prisma_1.default.menuOption.findUnique({
                            where: { id: detailOption.menuOptionId },
                        });
                        const option = await prisma_1.default.option.findUnique({
                            where: { id: detailOption.optionId },
                        });
                        return {
                            type: menuOption.name,
                            name: option.name,
                            price: option.price,
                        };
                    })),
                },
            };
        }));
        let referenceId = data.referenceId;
        let attempt = 0;
        const maxAttempts = 10;
        if (!referenceId) {
            if (attempt >= maxAttempts) {
                throw new Error('Unable to generate a unique reference ID after multiple attempts');
            }
            let exists;
            do {
                referenceId = (0, utils_1.generateReferenceId)();
                const res = await prisma_1.default.order.findUnique({
                    where: { referenceId },
                });
                exists = !!res;
                attempt++;
            } while (exists);
        }
        return prisma_1.default.order.create({
            data: {
                referenceId: referenceId,
                totalAmount: totalPrice,
                orderTime: new Date(data.orderTime),
                pickupTime: new Date(data.pickupTime),
                userId: data.userId,
                status: "pending",
                ...(data.transactionId && {
                    transactionId: data.transactionId,
                }),
                ...(data.vehicleId && {
                    userVehicleId: data.vehicleId,
                }),
                ...(data.merchantId && {
                    merchantId: data.merchantId,
                }),
                orderDetails: {
                    create: orderDetails,
                },
            },
            include: {
                orderDetails: {
                    include: {
                        orderDetailOptions: true,
                    },
                },
            },
        });
    }
    async findAll() {
        return prisma_1.default.order.findMany({
            include: {
                orderDetails: {
                    include: {
                        orderDetailOptions: true
                    }
                },
                userVehicle: true,
                transaction: true,
                rating: true,
                user: true
            }
        });
    }
    async findOne(id) {
        const order = await prisma_1.default.order.findUnique({
            where: { id },
            include: {
                orderDetails: {
                    include: {
                        orderDetailOptions: true,
                        menu: {
                            select: {
                                name: true,
                                image: true,
                            }
                        },
                    }
                },
                merchant: {
                    select: {
                        id: true,
                        name: true,
                        address: true
                    }
                },
                userVehicle: true,
                transaction: true,
                rating: true,
                user: true
            }
        });
        const restructuredOrder = {
            id: order.id,
            voucherUsed: order.voucherUsed,
            totalAmount: order.totalAmount,
            orderTime: order.orderTime,
            pickupTime: order.pickupTime,
            status: order.status,
            createdAt: order.createdAt,
            updatedAt: order.updatedAt,
            transactionId: order.transactionId,
            merchantId: order.merchant.id,
            merchantAddress: order.merchant.address,
            merchantName: order.merchant.name,
            orderDetails: order.orderDetails.map(orderDetail => ({
                id: orderDetail.id,
                quantity: orderDetail.quantity,
                basePrice: orderDetail.basePrice,
                createdAt: orderDetail.createdAt,
                updatedAt: orderDetail.updatedAt,
                orderId: orderDetail.orderId,
                menuId: orderDetail.menuId,
                menuName: orderDetail.menu?.name || null,
                menuImage: orderDetail.menu?.image || null,
                orderDetailOptions: orderDetail.orderDetailOptions,
            })),
            transaction: order.transaction,
            rating: order.rating,
            vehicle: order.userVehicle,
            user: order.user
        };
        return restructuredOrder;
    }
    async findByUserId(data) {
        const pageSize = data.numPerPage;
        const skip = (data.page - 1) * pageSize;
        const orders = await prisma_1.default.order.findMany({
            take: pageSize,
            skip: skip,
            where: { userId: data.userId },
            include: {
                orderDetails: {
                    include: {
                        orderDetailOptions: true,
                        menu: {
                            select: {
                                name: true,
                                image: true,
                            }
                        },
                    }
                },
                merchant: {
                    select: {
                        id: true,
                        name: true,
                        address: true
                    }
                },
                userVehicle: true,
                transaction: true,
                rating: true,
                user: true
            }
        });
        const restructuredOrders = orders.map(order => ({
            id: order.id,
            totalAmount: order.totalAmount,
            orderTime: order.orderTime,
            pickupTime: order.pickupTime,
            status: order.status,
            createdAt: order.createdAt,
            updatedAt: order.updatedAt,
            transactionId: order.transactionId,
            merchantId: order.merchant.id,
            merchantAddress: order.merchant.address,
            merchantName: order.merchant.name,
            orderDetails: order.orderDetails.map(orderDetail => ({
                id: orderDetail.id,
                quantity: orderDetail.quantity,
                basePrice: orderDetail.basePrice,
                createdAt: orderDetail.createdAt,
                updatedAt: orderDetail.updatedAt,
                orderId: orderDetail.orderId,
                menuId: orderDetail.menuId,
                menuName: orderDetail.menu?.name || null,
                menuImage: orderDetail.menu?.image || null,
                orderDetailOptions: orderDetail.orderDetailOptions,
            })),
            transaction: order.transaction,
            rating: order.rating,
            vehicle: order.userVehicle,
            user: order.user
        }));
        return restructuredOrders;
    }
    async findByMerchantId(data) {
        const pageSize = data.numPerPage;
        const skip = (data.page - 1) * pageSize;
        const [orders, totalCount] = await Promise.all([
            prisma_1.default.order.findMany({
                take: pageSize,
                skip: skip,
                where: {
                    merchantId: data.merchantId,
                    ...(data.status && {
                        status: data.status,
                    }),
                    ...(data.orderTime && {
                        orderTime: data.orderTime,
                    }),
                    ...(data.pickupTime && {
                        pickupTime: data.pickupTime,
                    }),
                    ...(data.userId && {
                        userId: data.userId,
                    }),
                },
                include: {
                    orderDetails: {
                        include: {
                            orderDetailOptions: true,
                            menu: {
                                select: {
                                    name: true,
                                    image: true,
                                },
                            },
                        },
                    },
                    merchant: {
                        select: {
                            id: true,
                            name: true,
                            address: true,
                        },
                    },
                    userVehicle: true,
                    transaction: true,
                    rating: true,
                    user: true,
                },
            }),
            prisma_1.default.order.count({
                where: {
                    merchantId: data.merchantId,
                    ...(data.status && {
                        status: data.status,
                    }),
                    ...(data.orderTime && {
                        orderTime: data.orderTime,
                    }),
                    ...(data.pickupTime && {
                        pickupTime: data.pickupTime,
                    }),
                    ...(data.userId && {
                        userId: data.userId,
                    }),
                },
            }),
        ]);
        const restructuredOrders = orders.map((order) => ({
            id: order.id,
            totalAmount: order.totalAmount,
            orderTime: order.orderTime,
            pickupTime: order.pickupTime,
            status: order.status,
            createdAt: order.createdAt,
            updatedAt: order.updatedAt,
            transactionId: order.transactionId,
            merchantId: order.merchant.id,
            merchantAddress: order.merchant.address,
            merchantName: order.merchant.name,
            orderDetails: order.orderDetails.map((orderDetail) => ({
                id: orderDetail.id,
                quantity: orderDetail.quantity,
                basePrice: orderDetail.basePrice,
                createdAt: orderDetail.createdAt,
                updatedAt: orderDetail.updatedAt,
                orderId: orderDetail.orderId,
                menuId: orderDetail.menuId,
                menuName: orderDetail.menu?.name || null,
                menuImage: orderDetail.menu?.image || null,
                orderDetailOptions: orderDetail.orderDetailOptions,
            })),
            transaction: order.transaction,
            rating: order.rating,
            vehicle: order.userVehicle,
            user: order.user,
        }));
        return { orders: restructuredOrders, totalCount };
    }
    async update(id, data) {
        return prisma_1.default.order.update({
            where: {
                id: id
            },
            data: {
                totalAmount: data.totalAmount,
                orderTime: new Date(),
                pickupTime: data.pickupTime,
                userId: data.userId,
                ...(data.transactionId && {
                    transactionId: data.transactionId
                }),
                ...(data.vehicleId && {
                    vehicleId: data.vehicleId,
                }),
                ...(data.merchantId && {
                    merchantId: data.merchantId
                }),
                orderDetails: {
                    createMany: {
                        data: data.orderDetails.map((elem) => ({
                            basePrice: 0,
                            quantity: elem.quantity,
                            menuId: elem.menuId,
                            orderDetailOption: {
                                create: elem.orderDetailOptions.map(option => ({
                                    type: "",
                                    name: "",
                                    price: 0,
                                })),
                            },
                        }))
                    },
                }
            }
        });
    }
    async updateMembershipTier(userId, newPoints) {
        const tiers = await prisma_1.default.tier.findMany({
            orderBy: { pointsRequired: 'asc' },
        });
        for (const tier of tiers) {
            if (newPoints >= tier.pointsRequired) {
                await prisma_1.default.user.update({
                    where: { id: userId },
                    data: { membershipTierId: tier.id },
                });
            }
        }
    }
    async updateStatus(id, data) {
        const order = await this.findOne(id);
        let pointsToAdd = 0;
        if (!order) {
            throw new Error('Order not found');
        }
        if (!order?.voucherUsed && data.status.toLocaleLowerCase() === "completed") {
            const user = await prisma_1.default.user.findUnique({
                where: { id: order.user.id },
                include: {
                    membershipTier: true,
                    loyaltyPoint: true
                },
            });
            const tier = await prisma_1.default.tier.findUnique({
                where: { id: user.membershipTierId },
            });
            pointsToAdd = Math.floor(order.totalAmount * tier.pointsMultiplier);
            const newAccumulatedPoints = user.loyaltyPoint.balance + pointsToAdd;
            await this.membershipService.addLoyaltyPoints(user.id, pointsToAdd, membership_dto_1.TransactionType.EARNED, order.id);
            await this.updateMembershipTier(user.id, newAccumulatedPoints);
        }
        return prisma_1.default.order.update({
            where: { id },
            data: {
                status: data.status,
                completedAt: data.status.trim().toLowerCase() === 'completed' ? new Date() : null,
            }
        });
    }
    async remove(id) {
        return prisma_1.default.order.delete({
            where: { id },
            include: {
                orderDetails: {
                    include: {
                        orderDetailOptions: true
                    }
                },
                transaction: true
            }
        });
    }
};
exports.OrderService = OrderService;
exports.OrderService = OrderService = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [membership_service_1.MembershipService])
], OrderService);
//# sourceMappingURL=order.service.js.map