import type { MultipleSchemas } from '../utils/schemaFileInput';
type PathType = 'file' | 'directory';
type SuccessfulLookupResult = {
    ok: true;
    schema: GetSchemaResult;
};
type NonFatalLookupError = {
    kind: 'NotFound';
    expectedType?: PathType;
    path: string;
} | {
    kind: 'WrongType';
    path: string;
    expectedTypes: PathType[];
} | {
    kind: 'FolderPreviewNotEnabled';
    path: string;
};
type LookupResult = SuccessfulLookupResult | {
    ok: false;
    error: NonFatalLookupError;
};
type PackageJsonLookupResult = SuccessfulLookupResult | {
    ok: false;
    error: {
        kind: 'PackageJsonNotConfigured';
    };
};
export type GetSchemaResult = {
    /**
     * A path from which schema was loaded
     * Can be either folder or a single file
     */
    schemaPath: string;
    /**
     * Base dir for all of the schema files.
     * In-multi file mode, this is equal to `schemaPath`.
     * In single-file mode, this is a parent directory of
     * a file
     */
    schemaRootDir: string;
    /**
     * All loaded schema files
     */
    schemas: MultipleSchemas;
};
export type GetSchemaOptions = {
    cwd?: string;
    argumentName?: string;
};
/**
 * Loads the schema, throws an error if it is not found
 * @param schemaPathFromArgs
 * @param opts
 */
export declare function getSchemaWithPath(schemaPathFromArgs?: string, { cwd, argumentName }?: GetSchemaOptions): Promise<GetSchemaResult>;
/**
 * Loads the schema, returns null if it is not found
 * Throws an error if schema is specified explicitly in
 * any of the available ways (argument, package.json config), but
 * can not be loaded
 * @param schemaPathFromArgs
 * @param param1
 * @returns
 */
export declare function getSchemaWithPathOptional(schemaPathFromArgs?: string, { cwd, argumentName }?: GetSchemaOptions): Promise<GetSchemaResult | null>;
export declare function readSchemaFromSingleFile(schemaPath: string): Promise<LookupResult>;
export type PrismaConfig = {
    schema?: string;
    seed?: string;
};
export declare function getPrismaConfigFromPackageJson(cwd: string): Promise<{
    data: PrismaConfig | undefined;
    packagePath: string;
} | null>;
export declare function getSchemaFromPackageJson(cwd: string): Promise<PackageJsonLookupResult>;
/**
 * Small helper that returns the directory which contains the `schema.prisma` file
 */
export declare function getSchemaDir(schemaPathFromArgs?: string): Promise<string | null>;
export declare function getSchema(schemaPathFromArgs?: string): Promise<MultipleSchemas>;
export {};
