"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FormReader = void 0;
const busboy_1 = __importDefault(require("busboy"));
const node_append_field_1 = __importDefault(require("../libs/node-append-field"));
const common_1 = require("@nestjs/common");
const file_type_1 = __importDefault(require("file-type"));
class FormReader {
    constructor(req, config) {
        this.req = req;
        this.config = config;
        this.fileStorePromises = [];
        this.result = {};
        this.files = [];
        this.busboy = (0, busboy_1.default)({
            headers: req.headers,
            limits: (config && config.limits) ? config.limits : {},
        });
        this.busboy.on('field', this.proceedField.bind(this));
        this.busboy.on('file', this.proceedFile.bind(this));
        this.busboy.on('error', this.rejectWithError.bind(this));
        this.busboy.on('partsLimit', () => this.rejectWithBadRequest(`Maximum number of parts is ${config.limits.parts}`));
        this.busboy.on('filesLimit', () => this.rejectWithBadRequest(`Maximum number of files is ${config.limits.files}`));
        this.busboy.on('fieldsLimit', () => this.rejectWithBadRequest(`Maximum number of fields is ${config.limits.fields}`));
        this.busboy.on('fileSize', () => this.rejectWithBadRequest(`Maximum file size is ${config.limits.fileSize}`));
        this.busboy.on('finish', this.proceedFinish.bind(this));
    }
    handle() {
        return new Promise((resolve, reject) => {
            this.handlePromiseReject = reject;
            this.handlePromiseResolve = resolve;
            this.req.pipe(this.busboy);
        });
    }
    async deleteFiles() {
        await Promise.all(this.files.map(f => (f === null || f === void 0 ? void 0 : f.delete()) || Promise.resolve()));
    }
    proceedField(fieldName, value, fieldNameTruncated, valueTruncated) {
        (0, node_append_field_1.default)(this.result, fieldName, value);
    }
    proceedFile(fieldName, fileStream, info) {
        const { filename, encoding, mimeType } = info;
        if (!filename) {
            fileStream.resume();
            return;
        }
        const readFilePromise = this.loadFile(filename, encoding, mimeType, fileStream)
            .then(f => {
            if (fileStream.truncated) {
                this.busboy.emit('fileSize');
            }
            else {
                this.files.push(f);
                (0, node_append_field_1.default)(this.result, fieldName, f);
            }
        }).catch(err => {
            this.rejectWithError(err);
        });
        this.fileStorePromises.push(readFilePromise);
    }
    async proceedFinish() {
        await Promise.all(this.fileStorePromises);
        this.handleDone();
        this.handlePromiseResolve(this.result);
    }
    rejectWithBadRequest(message) {
        const error = new common_1.BadRequestException(null, message);
        this.handlePromiseReject(error);
        this.handleDone();
    }
    handleDone() {
        this.req.unpipe(this.busboy);
        this.busboy.removeAllListeners();
    }
    rejectWithError(err) {
        if ((err === null || err === void 0 ? void 0 : err.message) === 'Unexpected end of form') {
            this.rejectWithBadRequest(err.message);
            return;
        }
        this.handlePromiseReject(err);
        this.handleDone();
    }
    async loadFile(originalName, encoding, mimetype, stream) {
        const streamWithFileType = await file_type_1.default.stream(stream);
        const storedFile = await this.config['storage'].create({
            originalName,
            encoding,
            mimetype,
        }, streamWithFileType, this.config);
        storedFile.setFileTypeResult(streamWithFileType.fileType);
        return storedFile;
    }
}
exports.FormReader = FormReader;
//# sourceMappingURL=FormReader.js.map