"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HasMimeType = void 0;
const class_validator_1 = require("class-validator");
const is_file_validator_1 = require("./is-file.validator");
const toArray_1 = require("../../helpers/toArray");
function HasMimeType(allowedMimeTypes, strictSource, validationOptions) {
    return (0, class_validator_1.ValidateBy)({
        name: 'HasMimeType',
        constraints: [allowedMimeTypes],
        validator: {
            validate(value, args) {
                const allowedMimeTypes = (0, toArray_1.toArray)(args.constraints[0]) || [];
                const strictSource = (typeof args.constraints[1] === 'string')
                    ? args.constraints[1]
                    : undefined;
                if ((0, is_file_validator_1.isFile)(value)) {
                    const mimeWithSource = value.mimeTypeWithSource;
                    const hasSourceMatch = !strictSource || strictSource === mimeWithSource.source;
                    if (!hasSourceMatch) {
                        return false;
                    }
                    for (let mimeType of allowedMimeTypes) {
                        switch (true) {
                            case typeof mimeType === 'string' && !mimeType.includes('*'):
                                if (mimeType === mimeWithSource.value) {
                                    return true;
                                }
                                break;
                            case typeof mimeType === 'string' && mimeType.includes('*'):
                                const regex = new RegExp(`^${mimeType}$`.replace('*', '.+'));
                                if (regex.test(mimeWithSource.value)) {
                                    return true;
                                }
                                break;
                            case mimeType instanceof RegExp:
                                if (mimeType.test(mimeWithSource.value)) {
                                    return true;
                                }
                                break;
                            default:
                                throw new Error(`Unknown mime type for validate`);
                        }
                    }
                }
                return false;
            },
            defaultMessage(validationArguments) {
                const allowedMimeTypes = (0, toArray_1.toArray)(validationArguments.constraints[0]) || [];
                return `File must be of one of the types ${allowedMimeTypes.join(', ')}`;
            },
        },
    }, validationOptions || ((typeof strictSource === 'object') ? strictSource : null));
}
exports.HasMimeType = HasMimeType;
//# sourceMappingURL=has-mime-type.validator.js.map