"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.writeToFile = exports.writeFromDMMF = exports.makeRootOutputFromClasses = exports.makeOutputFromPrismaClass = exports.swaggerOptionToTemplate = exports.convertDMMFModelToClass = exports.convertField = void 0;
const change_case_1 = require("change-case");
const path = require("path");
const fs = require("fs");
const templates_1 = require("./templates");
const convertField = (field) => {
    const result = {
        swaggerOption: {},
        name: field.name,
    };
    const mapType = {
        Int: "number",
        String: "string",
        DateTime: "Date",
        Boolean: "boolean",
        Json: "JSON",
    };
    let type = mapType[field.type];
    if (type) {
        result.type = type;
        result.swaggerOption.type = type.charAt(0).toUpperCase() + type.slice(1);
        return result;
    }
    type = field.type;
    if (field.isList) {
        result.type = type += "[]";
        result.swaggerOption.isArray = true;
    }
    if (field.relationName) {
        result.swaggerOption.type = `() => _${field.type}`;
        result.type = `_${type}`;
        return result;
    }
    if (field.kind === "enum") {
        result.swaggerOption.enum = field.type;
        result.swaggerOption.enumName = `'${field.type}'`;
        result.type = field.type;
    }
    return result;
};
exports.convertField = convertField;
const convertDMMFModelToClass = (model) => {
    const relationTypes = model.fields
        .filter((field) => field.relationName && model.name !== field.type)
        .map((v) => v.type);
    const enums = model.fields.filter((field) => field.kind === "enum");
    return {
        name: model.name,
        fields: model.fields.map((field) => exports.convertField(field)),
        relationTypes: [...new Set(relationTypes)],
        enumTypes: enums.map((field) => field.type),
    };
};
exports.convertDMMFModelToClass = convertDMMFModelToClass;
const swaggerOptionToTemplate = (input) => {
    const rows = Object.entries(input).reduce((result, [k, v]) => {
        result.push(`${k}:${v}`);
        return result;
    }, []);
    return `@ApiProperty({${rows.join(", ")}})`;
};
exports.swaggerOptionToTemplate = swaggerOptionToTemplate;
const makeOutputFromPrismaClass = (pClass, useSwagger = false) => {
    const fieldContent = pClass.fields.map((_field) => templates_1.fieldTemplates
        .replace("#!{NAME}", _field.name)
        .replace("#!{TYPE}", _field.type)
        .replace("#!{DECORATORS}", useSwagger ? exports.swaggerOptionToTemplate(_field.swaggerOption) : ""));
    const toImports = pClass.relationTypes.map((relationClassName) => `import { _${change_case_1.pascalCase(relationClassName)} } from './${change_case_1.snakeCase(relationClassName)}'`);
    if (useSwagger) {
        toImports.push(`import { ApiProperty } from '@nestjs/swagger'`);
    }
    const enums = pClass.enumTypes.map((enumName) => `import { ${enumName} } from '@prisma/client'`);
    return templates_1.classTemplates
        .replace("#!{IMPORTS}", [...toImports, ...enums].join("\r\n"))
        .replace("#!{NAME}", `_${pClass.name}`)
        .replace("#!{FIELDS}", fieldContent.join("\r\n"));
};
exports.makeOutputFromPrismaClass = makeOutputFromPrismaClass;
const makeRootOutputFromClasses = (classes) => {
    return templates_1.rootTemplates
        .replace("#!{IMPORTS}", `${classes
        .map((_class) => `import { _${change_case_1.pascalCase(_class.name)} } from "./${change_case_1.snakeCase(_class.name)}"`)
        .join("\r\n")}`)
        .replace("#!{CLASSES}", `${classes
        .map((v) => `export class ${change_case_1.pascalCase(v.name)} extends _${change_case_1.pascalCase(v.name)} {}`)
        .join(`\r\n`)}`)
        .replace("#!{CLASSE_NAMES}", `[${classes.map((v) => change_case_1.pascalCase(v.name)).join(`, `)}]`);
};
exports.makeRootOutputFromClasses = makeRootOutputFromClasses;
const writeFromDMMF = (input) => {
    const { dmmf, outputType, targetDir, useSwagger = false } = input;
    const models = dmmf.datamodel.models;
    const classes = models.map((model) => exports.convertDMMFModelToClass(model));
    classes.forEach((_class) => {
        const classOutput = exports.makeOutputFromPrismaClass(_class, useSwagger);
        if (outputType === "console") {
            console.log(classOutput);
            return;
        }
        exports.writeToFile({
            dirPath: targetDir,
            fileName: `${change_case_1.snakeCase(_class.name)}.ts`,
            content: classOutput,
        });
    });
    if (outputType === "file") {
        exports.writeToFile({
            dirPath: targetDir,
            fileName: `model.ts`,
            content: exports.makeRootOutputFromClasses(classes),
        });
    }
    else {
        console.log(exports.makeRootOutputFromClasses(classes));
    }
};
exports.writeFromDMMF = writeFromDMMF;
const writeToFile = (input) => {
    const { dirPath, fileName, content } = input;
    const targetDirPath = path.resolve(dirPath, "_gen");
    if (fs.existsSync(targetDirPath) === false) {
        fs.mkdirSync(targetDirPath, { recursive: true });
    }
    const filePath = path.resolve(targetDirPath, fileName);
    console.log(`write to ${filePath}..`);
    fs.writeFileSync(filePath, content);
};
exports.writeToFile = writeToFile;
//# sourceMappingURL=prismaToClass.js.map