// seed.ts
import { PrismaClient } from '@prisma/client';

import SupabaseService from '../src/auth/supabase.service';
import { sampleData } from './mock/data';
import { randomBytes } from 'crypto';

const prisma = new PrismaClient();
const supabaseService = new SupabaseService();

async function seed() {
    try {

        // ******** Warning .. Do not run this seed on production ********* //

        // Enable to reset DB
        let resetDb = false;

        let updateReferenceIds = true;

        if (resetDb) {

            await resetDatabase();
        }

        if (updateReferenceIds) {

            await updateExistingOrders();
        }


        console.log('Seed data created successfully');
    } catch (error) {
        console.error('Error seeding data:', error);
    } finally {
        await prisma.$disconnect();
    }
}

async function resetDatabase() {

    //independent
    //relation
    //order details, transaction, and order need to delete first
    await prisma.orderDetailOptions.deleteMany();
    await prisma.orderDetails.deleteMany();
    await prisma.transaction.deleteMany();
    await prisma.order.deleteMany();

    //then option
    await prisma.option.deleteMany();
    //then menu option
    await prisma.menuOption.deleteMany();
    // then menu 
    await prisma.menu.deleteMany();
    // then merchant
    await prisma.merchant.deleteMany();

    await prisma.user.deleteMany();

    // after all delete setting can be deleted
    await prisma.role.deleteMany();
    await prisma.tier.deleteMany();

    // Create users
    for (const role of sampleData.roles) {
        await prisma.role.create({
            data: {
                id: role.id,
                name: role.name
            }
        });
    }

    for (const data of sampleData.users) {
        await supabaseService.deleteUserFromAuth(data.email);

        await supabaseService.signUpWithEmail(
            data.email,
            data.password,
        );

        await prisma.user.create({
            data: {
                id: data.id,
                email: data.email,
                name: data.name,
                phoneNumber: data.phoneNumber,
                dateOfBirth: new Date(data.dateOfBirth),
                role: {
                    connect: {
                        id: data.roleId
                    }
                },
                address: {
                    ...(data.address)
                }
            }
        });
    }

    for (const data of sampleData.merchants) {
        await supabaseService.deleteUserFromAuth(data.email);

        await supabaseService.signUpWithEmail(
            data.email,
            data.password,
        );

        await prisma.user.create({
            data: {
                id: data.id,
                email: data.email,
                name: data.name,
                phoneNumber: data.phoneNumber,
                dateOfBirth: new Date(data.dateOfBirth),
                role: {
                    connect: {
                        id: data.roleId
                    }
                },
                merchant: {
                    ...(data.merchant)
                }
            }
        });
    }

    var menuid = "";
    for (const menuData of sampleData.menus) {
        const createdMenu = await prisma.menu.create({
            data: {
                name: menuData.name,
                basePrice: menuData.basePrice,
                category: menuData.category,
                description: menuData.description,
                merchant: { connect: { id: menuData.merchantId } },
                image: menuData.image
            },
        });
        menuid = createdMenu.id;

        for (const menuOptionData of menuData.menuOptions) {
            const createdMenuOption = await prisma.menuOption.create({
                data: {
                    name: menuOptionData.name,
                    isRequired: menuOptionData.isRequired,
                    min: menuOptionData.min,
                    max: menuOptionData.max,
                    menu: { connect: { id: createdMenu.id } },
                },
            });

            for (const optionData of menuOptionData.options) {
                await prisma.option.create({
                    data: {
                        name: optionData.name,
                        price: optionData.price,
                        menuOption: { connect: { id: createdMenuOption.id } },
                    },
                });
            }
        }
    }

    for (const orderData of sampleData.orders) {
        const order = await prisma.order.create({
            data: {
                ...orderData,
                referenceId:  generateReferenceId(),
                orderDetails: {
                    create: orderData.orderDetails.map((orderDetail) => ({
                        quantity: orderDetail.quantity,
                        basePrice: orderDetail.basePrice,
                        orderDetailOptions: {
                            create: orderDetail.orderDetailOptions.map((orderDetailOption) => ({
                                type: orderDetailOption.type,
                                name: orderDetailOption.name,
                                price: orderDetailOption.price,
                            })),
                        },
                        menu: { connect: { id: menuid } }, // Connect to the appropriate menu
                    })),
                },
            },
        });

        for (const transactionsData of sampleData.transactions) {
            await prisma.transaction.create({
                data: {
                    ...transactionsData,
                    orderId: order.id,
                },
            });
        }
    }

    // Credit Data
    for (const creditsData of sampleData.credit) {

        const credit = await prisma.credit.findUnique({
            where: {
                id: creditsData.id
            }
        })

        if (!credit) {
            await prisma.credit.create({
                data: {
                    ...creditsData,
                },
            });
        }
    }

    // Credit Transaction Data
    for (const creditTransactionsData of sampleData.creditHistory) {

        const creditTransaction = await prisma.creditHistory.findUnique({
            where: {
                id: creditTransactionsData.id
            }
        })

        if (!creditTransaction) {
            await prisma.creditHistory.create({
                data: {
                    ...creditTransactionsData,
                },
            });
        }
    }

    // loyality configuration
    await prisma.tier.createMany({
        data: [
            { name: 'Silver', pointsRequired: 0, pointsMultiplier: 1.0 },
            { name: 'Gold', pointsRequired: 600, pointsMultiplier: 1.5 },
            { name: 'Platinum', pointsRequired: 1000, pointsMultiplier: 2.0 },
        ],
    });

    // Find the Silver tier
    const silverTier = await prisma.tier.findUnique({
        where: { name: 'Silver' },
    });

    if (!silverTier) {
        throw new Error('Silver tier not found');
    }

    // Update all existing users to have the Silver tier
    await prisma.user.updateMany({
        data: {
            membershipTierId: silverTier.id,
        },
    });

    console.log('All users have been updated to Silver tier');
}

function generateReferenceId() {

    const now = new Date();
    const year = now.getFullYear().toString().slice(-2); // Last two digits of the year
    const month = String(now.getMonth() + 1).padStart(2, '0'); // MM
    const day = String(now.getDate()).padStart(2, '0'); // DD
    const datePrefix = `${year}${month}${day}`; // YYMMDD

    // Generate a shorter random part (4 bytes = 8 hex characters)
    const randomSuffix = randomBytes(4).toString('hex'); // 4 bytes = 8 hex characters

    return `ORD-${datePrefix}-${randomSuffix}`;
}

async function updateExistingOrders() {
    const ordersWithoutRefId = await prisma.order.findMany({
        where: { referenceId: null },
    });

    for (const order of ordersWithoutRefId) {
        let referenceId: string;
        let exists: boolean;

        do {
            // Generate a unique referenceId (e.g., using UUID)
            referenceId = generateReferenceId()// or use a custom function

            // Check if the referenceId already exists
            const res = await prisma.order.findUnique({
                where: { referenceId },
            });

            exists = !!res; // Set exists to true if a record is found, otherwise false
        } while (exists);

        // Update the order with the new referenceId
        await prisma.order.update({
            where: { id: order.id },
            data: { referenceId },
        });

        console.log(`Updated Order ID ${order.id} with Reference ID ${referenceId}`);
    }
}

seed();
