// src/menu/menu.controller.ts

import {
  Body,
  Controller,
  HttpCode,
  HttpException,
  HttpStatus,
  Post,
} from '@nestjs/common';

import { Platform } from '../types';
import { UserService } from '../user/user.service';
import {
  SignInDto,
  SignUpCustomerDto,
  SignUpMerchantDto,
  ResetPasswordDto,
  EmailResetPasswordDto,
  PasswordResetOtpDto,
  SetNewPasswordDto
} from './auth.dto';
import { AuthService } from './auth.service';
import { SupabaseService } from './supabase.service';

@Controller('auth')
export class AuthController {
  constructor(
    private authService: AuthService,
    private userService: UserService,
    private supabaseService: SupabaseService,
  ) { }

  @Post('user/signup')
  @HttpCode(HttpStatus.OK)
  async userSignUp(@Body() data: SignUpCustomerDto) {
    const { email, password } = data;
    
    let existinguser = await this.authService.findUserByEmail(email);

    if (existinguser) {
      throw new HttpException({ error: 'This email is existed.' }, HttpStatus.BAD_REQUEST);
    }

    const { data: authData, error } = await this.supabaseService.signUpWithEmail(
      email,
      password,
    );

    if (error) {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }

    await this.authService.createCustomer(data);

    return { data: authData, message: 'Customer successfully created' };
  }

  @Post('merchant/signup')
  @HttpCode(HttpStatus.OK)
  async merchantSignUp(@Body() data: SignUpMerchantDto) {
    const { email, password } = data;

    let existinguser = await this.authService.findUserByEmail(email);

    if (existinguser) {
      throw new HttpException({ error: 'This email is existed.' }, HttpStatus.BAD_REQUEST);
    }

    const { data: authData, error } = await this.supabaseService.signUpWithEmail(
      email,
      password,
    );

    if (error) {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }

    await this.authService.createMerchant(data);

    return { data: authData, message: 'Merchant successfully created' };
  }

  @Post('login')
  @HttpCode(HttpStatus.OK)
  async login(@Body() userData: SignInDto) {
    const { email, password, platform } = userData;

    if (!platform) {
      throw new HttpException({ error: 'Missing Platform' }, HttpStatus.BAD_REQUEST);
    }

    let existingUser = await this.authService.findUserByEmail(email);

    if (!existingUser) {
      throw new HttpException({ error: 'This user does not exist in our system' }, HttpStatus.BAD_REQUEST);
    }

    if (platform === Platform.Customer) {
      if (existingUser.role.name !== Platform.Customer) {
        throw new HttpException({ error: 'This role should not login to Customer Portal' }, HttpStatus.BAD_REQUEST);
      }
    } else {
      if (existingUser.role.name !== Platform.Merchant) {
        throw new HttpException({ error: 'This role should not login to Merchant Portal' }, HttpStatus.BAD_REQUEST);
      }
    }

    if (existingUser) {
      const { data, error } = await this.supabaseService.signInWithEmail(
        email,
        password,
      );

      if (data) {
        data['userData'] = await this.userService.findOneByEmail(email);
        return { data: data, message: 'Login Successfully.' };
      } else {
        throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
      }
    } else {
      throw new HttpException({ error: 'User Not Found' }, HttpStatus.BAD_REQUEST);
    }
  }

  @Post('signout')
  @HttpCode(HttpStatus.OK)
  async signOut() {
    const { error } = await this.supabaseService.signOut();

    if (!error) {
      return { data: {}, message: 'Sign Out Successfully.' };
    } else {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }
  }

  @Post('resetpassword')
  @HttpCode(HttpStatus.OK)
  async resetPassword(@Body() body: ResetPasswordDto) {
    const { error } = await this.supabaseService.ResetPassword(body);

    if (!error) {
      await this.supabaseService.signOut();
      return { data: {}, message: 'Reset Password Successfully.' };
    } else {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }
  }

  @Post('sentresetotp')
  @HttpCode(HttpStatus.OK)
  async sentResetPasswordOtpToEmail(@Body() body: EmailResetPasswordDto) {
    const { error } = await this.supabaseService.sentResetPasswordEmail(body.email);

    if (!error) {
      return { data: {}, message: 'Reset Password Otp Email Sent Successfully.' };
    } else {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }
  }

  @Post('verifyotp')
  @HttpCode(HttpStatus.OK)
  async verifOpt(@Body() body: PasswordResetOtpDto) {
    const { error } = await this.supabaseService.verifyResetPasswordOtp(body.email, body.passwordResetOtp);

    if (!error) {
      return { data: {}, message: 'Verify Otp Successfully.' };
    } else {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }
  }

  @Post('setnewpassword')
  @HttpCode(HttpStatus.OK)
  async setNewPassword(@Body() body: SetNewPasswordDto) {
    const { error } = await this.supabaseService.setNewPassword(body);

    if (!error) {
      return { data: {}, message: 'Reset Password Successfully.' };
    } else {
      throw new HttpException({ error: error['message'] }, HttpStatus.BAD_REQUEST);
    }
  }

}
