import { Injectable } from '@nestjs/common';
import {
  createClient,
  SupabaseClient,
} from '@supabase/supabase-js';
import {
  ResetPasswordDto,
  SetNewPasswordDto
} from './auth.dto';

import prisma from '../prisma';

@Injectable()
export class SupabaseService {
  private supabase: SupabaseClient;

  constructor() {
    this.supabase = createClient(
      process.env.SUPABASE_URL,
      process.env.DATABASE_PUBLIC_KEY,
    );
  }

  async signUpWithEmail(email: string, password: string) {
    try {
      // Sign up a user with email using Supabase
      const { data, error } = await this.supabase.auth.signUp({
        email,
        password: password,
      });

      if (error) {
        return { error };
      }

      return {
        data,
        message: 'Verification email sent. Please check your inbox.',
      };
    } catch (error) {
      return { error: 'An error occurred during sign-up.' };
    }
  }

  async verifyOTP(email: string, otp: string) {
    try {
      // Verify the OTP for a user's email using Supabase
      const { data, error } = await this.supabase.auth.verifyOtp({
        email,
        token: otp,
        type: 'email',
      });

      if (error) {
        return { error };
      }

      return {
        data,
        message: 'OTP verified successfully.',
      };
    } catch (error) {
      return { error: 'An error occurred during OTP verification.' };
    }
  }

  async signInWithEmail(email: string, password: string) {
    try {
      const { data, error } = await this.supabase.auth.signInWithPassword({
        email: email,
        password: password,
      });

      if (error) {
        return { error };
      }

      return {
        data,
        message: 'Login successfully.',
      };
    } catch (error) {
      return { error: 'Login unsuccessfully.' };
    }
  }

  async deleteUserFromAuth(email: string) {
    await this.supabase.from('auth.users').delete().match({ email: email });
  }

  async validateUserSession(token: string) {
    await this.supabase.auth.getUser(token);
  }

  async signOut() {
    try {
      const currentUser = this.supabase.auth.getSession();

      if (currentUser) {
        const { error } = await this.supabase.auth.signOut();

        if (error) {
          return { error };
        }

        return { message: 'Logout successful.' };
      } else {
        return { error: 'You do not have permission to sign out this user.' };
      }
    } catch (error) {
      return { error: 'Logout unsuccessful.' };
    }
  }

  async ResetPassword(password: ResetPasswordDto ) {
    try {
      const currentUser = await this.supabase.auth.getUser();
      
      const { data, error } = await this.supabase.auth.signInWithPassword({
        email: currentUser.data.user.email,
        password: password.currentPassword,
      });

      if(error){
        return { message: 'Current Password Not Match.' };
      }

      if (currentUser) {

        if(password.newPassword !== password.confirmNewPassword){
          return { message: 'New Password Not Match.' };
        }
        
        const { data, error } = await this.supabase.auth.updateUser({password: password.newPassword})

        if (error) {
          return { error };
        }

        return { message: 'Password reset successful.' };
      } else {
        return { error: 'You do not have permission to sign out this user.' };
      }
    } catch (error) {
      return { error: 'Password reset unsuccessful.' };
    }
  }

  async sentResetPasswordEmail(email:string){
    try {
      const existingUser = await prisma.user.findUnique({
        where:{
          email,
        }
      });

      if (existingUser) {
        const { data, error } = await this.supabase.auth.resetPasswordForEmail(email)

        if(error){
          return { error };
        }

        return { message: 'Password reset email sent successful.' };
      } else {
        return { message: 'Email Not Found.' };
      }
    } catch (error) {
      return { error: 'Password reset email sent unsuccessful.' };
    }
  }

  async verifyResetPasswordOtp(email: string, otp: string){
    try {

      const { data, error } = await this.supabase.auth.verifyOtp({
        email,
        token: otp,
        type: 'email',
      });

      if (error) {
        return { error };
      }

      return {
        data,
        message: 'OTP verified successfully.',
      };
    } catch (error) {
      return { error: 'OTP verified unsuccessful.' };
    }
  }

  async setNewPassword(password: SetNewPasswordDto){
    try {
      const currentUser = await this.supabase.auth.getUser();
      
      if (currentUser) {
        if(password.newPassword !== password.confirmNewPassword){
          return { message: 'New Password Not Match.' };
        }
        
        const { data, error } = await this.supabase.auth.updateUser({password: password.newPassword})

        if (error) {
          return { error };
        }

        return { message: 'Password reset successful.' };
      } else {
        return { error: 'You do not have permission to sign out this user.' };
      }
    } catch (error) {
      return { error: 'Password reset unsuccessful.' };
    }
  }
}

export default SupabaseService;
