import { Injectable } from '@nestjs/common';
import { User } from '@prisma/client';
import { BillplzCallbackDto } from 'src/payment/payment.dto';
import prisma from '../prisma';
import { PaymentType } from './credit.dto';

@Injectable()
export class CreditService {

    constructor() { }

    async getUserBalance(userId: string) {

        return await prisma.credit.findFirst({
            where: { userId: userId },
            include: {
                history: {
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 5,
                }
            }
        })
    }

    async create(user: User) {

        const newCredit = await prisma.credit.create({
            data: {
                balance: 0.00,
                userId: user.id,
                createdAt: new Date(),
                user: { connect: { id: user.id } },
            },
        })

        // Update the user with the new creditId
        await prisma.user.update({
            where: { id: user.id },
            data: { creditId: newCredit.id },
        });

        // Fetch the updated user
        return await this.getUserBalance(user.id)
    }

    async addCredit(callBackData: BillplzCallbackDto) {

        const billplz = await this.createBillPlzRecords(callBackData)

        const amount = +callBackData.amount / 100;

        // Find user by email
        let user = await prisma.user.findUnique({
            where: { email: callBackData.email }
        })

        // Fetch the user's credit record
        let credit = await prisma.credit.findUnique({
            where: { userId: user.id },
        });

        // If the user doesn't have a credit record, create one
        if (!credit) {
            credit = await prisma.credit.create({
                data: {
                    userId: user.id,
                    balance: 0,
                    createdAt: new Date(),
                    user: { connect: { id: user.id } },
                },
            });

            // Update the user with the new creditId
            await prisma.user.update({
                where: { id: user.id },
                data: { creditId: credit.id },
            });
        }

        // Update the credit balance
        const updatedCredit = await prisma.credit.update({
            where: { userId: user.id },
            data: {
                balance: credit.balance + amount,
                updatedAt: new Date(),
            },
        });

        // Update CreditHistory based on the Billplz.id
        const creditHistory = await prisma.creditHistory.findFirst({
            where: { billplzId: billplz.id },
        });

        if (creditHistory) {
            // Update existing CreditHistory
            await prisma.creditHistory.update({
                where: { id: creditHistory.id },
                data: {
                    actualAmount: billplz.paid_amount,
                    type: PaymentType.RELOAD,
                    // status: Status.SUCCESS,
                    updatedAt: new Date(),
                    lastCreditAmount: credit.balance
                },
            });
        } else {
            // Create new CreditHistory entry
            await prisma.creditHistory.create({
                data: {
                    amount: billplz.amount,
                    actualAmount: billplz.paid_amount,
                    type: PaymentType.RELOAD,
                    // orderId: orderId, // Replace with actual order ID
                    creditId: updatedCredit.id,
                    lastCreditAmount: credit.balance,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                    billplzId: billplz.id,
                    status: billplz.state.toUpperCase()
                },
            });
        }


        return updatedCredit;
    }

    async createBillPlzRecords(callBackData: BillplzCallbackDto) {

        // Create a new Billplz record
        const billplz = await prisma.billplz.create({
            data: {
                billplz_id: callBackData.id,
                collection_id: callBackData.collection_id,
                paid: (callBackData.paid === "true"),
                state: callBackData.state,
                amount: +callBackData.amount / 100, // Convert to standard amount format
                paid_amount: +callBackData.paid_amount / 100, // Convert to standard amount format
                due_at: callBackData.due_at,
                email: callBackData.email,
                mobile: callBackData?.mobile,
                name: callBackData.name,
                url: callBackData.url,
                paid_at: new Date(callBackData.paid_at),
                x_signature: callBackData.x_signature,
            },
        });

        return billplz
    }
}
