import { Body, Controller, Delete, Get, HttpCode, HttpStatus, Param, Post, Put, UploadedFile, UseInterceptors } from '@nestjs/common';
import { FileInterceptor } from '@nestjs/platform-express';
import { ApiTags } from '@nestjs/swagger';
import { UserService } from 'src/user/user.service';
import { isEmpty } from 'src/utils';
import { CreateProductDto, ListProductsDto, LoyaltyPointHistoryDto, MarkAsCollectedDto, RedeemProductDto, UpdateProductDto, UserRedemptionDto } from './membership.dto';
import { MembershipService } from './membership.service';

@ApiTags('Membership')
@Controller('membership')
export class MembershipController {

    constructor(
        private membershipService: MembershipService,
        private userService: UserService
    ) { }

    @Get('/point/:id')
    @HttpCode(HttpStatus.OK)
    async findBalanceByUserId(@Param('id') userId: string) {

        const membership = await this.membershipService.getLoyaltyPointBalance(userId);
        const user = await this.userService.findOne(userId);

        // check if data is empty, create new membership for user.
        if (isEmpty(membership) && !isEmpty(user)) {

            console.log(`Found no registered Membership User & User is active user! Registering new Membership for ${user?.name}.`)
            return { data: await this.membershipService.create(user), message: 'Find Membership Loyalty Point Balance' }
        }

        return { data: membership, message: 'Find Membership Loyalty Point Balance' }
    }

    @Get(':userId')
    getMembershipDetails(@Param('userId') userId: string) {
        return this.membershipService.getMembershipDetails(userId);
    }

    @Post('/point/history')
    @HttpCode(HttpStatus.OK)
    async findCreditHistory(@Body() loyaltyPointHistoryDto: LoyaltyPointHistoryDto) {

        const creditHistory = await this.membershipService.getUserLoyaltyPointHistory(loyaltyPointHistoryDto);

        return { data: creditHistory, message: 'Find Loyalty Point History' }
    }

    // todo add products, view products, edit products, purchase with loyalty point
    @Post('/merchandise/create')
    @UseInterceptors(FileInterceptor('imagefile', { limits: { fileSize: 5 * 1024 * 1024 } }))  // 5MB limit
    async createProduct(@UploadedFile() image: Express.Multer.File, @Body() createProductDto: CreateProductDto) {

        let intake = {
            ...createProductDto,
            stock: typeof createProductDto.stock === 'string' ? parseFloat(createProductDto.stock) : createProductDto.stock,
            price: typeof createProductDto.price === 'string' ? parseFloat(createProductDto.price) : createProductDto.price,
            pointsCost: typeof createProductDto.pointsCost === 'string' ? parseFloat(createProductDto.pointsCost) : createProductDto.pointsCost,
        }

        let data = await this.membershipService.createProduct(image, intake)

        return { data: data, message: 'Create Merchandise Product' };
    }

    @Put('/merchandise/update/:id')
    @UseInterceptors(FileInterceptor('imagefile'))
    async updateProduct(@Param('id') id: string, @UploadedFile() image: Express.Multer.File, @Body() updateProductDto: UpdateProductDto) {

        let intake = {
            ...updateProductDto,
            stock: typeof updateProductDto.stock === 'string' ? parseFloat(updateProductDto.stock) : updateProductDto.stock,
            price: typeof updateProductDto.price === 'string' ? parseFloat(updateProductDto.price) : updateProductDto.price,
            pointsCost: typeof updateProductDto.pointsCost === 'string' ? parseFloat(updateProductDto.pointsCost) : updateProductDto.pointsCost,
        }

        return { data: await this.membershipService.updateProduct(id, image, intake), message: 'Update Merchandise Product' };
    }

    @Delete('/merchandise/delete/:id')
    async deleteProduct(@Param('id') id: string) {

        const res = await this.membershipService.deleteProduct(id);

        return { data: res, message: 'Delete Merchandise Product' };
    }

    @Post('/merchandise/list')
    async listProducts(@Body() listProductsDto: ListProductsDto) {

        return { data: await this.membershipService.listProducts(listProductsDto), message: 'Get Merchandise Product List' };
    }

    @Post('/merchandise/redeem')
    async redeemProduct(@Body() redeemProductDto: RedeemProductDto) {

        return { data: await this.membershipService.redeemProduct(redeemProductDto), message: 'Redeem Merchandise' };
    }

    @Put('/merchandise/collect/:id')
    async markAsCollected(@Param('id') id: string, @Body() markAsCollectedDto: MarkAsCollectedDto) {

        markAsCollectedDto.id = id;
        return { data: await this.membershipService.markAsCollected(markAsCollectedDto), message: 'Update Redempted Merchandise Product' };
    }

    @Post('/merchandise/user-redemption')
    @HttpCode(HttpStatus.OK)
    async findUserRedemptionHistory(@Body() userRedemptionDto: UserRedemptionDto) {

        const userRedemption = await this.membershipService.getUserRedemptionHistory(userRedemptionDto);
        return { data: userRedemption, message: 'Find User Redemption History' }
    }

}
