import { Injectable, InternalServerErrorException, NotFoundException } from '@nestjs/common';
import { User } from '@prisma/client';
import { isEmpty, UploadImg } from 'src/utils';
import prisma from '../prisma';
import { DeliveryMethod, ListProductsDto, LoyaltyPointHistoryDto, MarkAsCollectedDto, RedeemProductDto, TransactionType, UserRedemptionDto } from './membership.dto';

const UploadFunction = new UploadImg();
@Injectable()
export class MembershipService {
    async markAsCollected(markAsCollectedDto: MarkAsCollectedDto) {

        const { id, courierType, shippingNo, remarks, collected } = markAsCollectedDto;

        const redemption = await prisma.userRedemptionHistory.findUnique({
            where: { id },
        });

        if (!redemption) {

            throw new NotFoundException({
                error: 'Redemption not found. Please try again later.',
                message: ['Redemption not found. Please try again later.'],
                statusCode: 500
            })
        }

        return await prisma.userRedemptionHistory.update({
            where: { id },
            data: {
                collected,
                courierType,
                shippingNo,
                remarks,
            },
        });
    }

    async getUserLoyaltyPointHistory(loyaltyPointHistoryDto: LoyaltyPointHistoryDto) {

        const whereClause: any = {};

        if (loyaltyPointHistoryDto.startDate) {

            whereClause.createdAt = {
                ...whereClause.createdAt,
                gte: new Date(loyaltyPointHistoryDto.startDate),
            };
        }

        if (loyaltyPointHistoryDto.endDate) {

            whereClause.createdAt = {
                ...whereClause.createdAt,
                lte: new Date(loyaltyPointHistoryDto.endDate),
            };
        }

        const loyaltyPointHistory = await prisma.loyaltyPoint.findUnique({
            where: {
                userId: loyaltyPointHistoryDto.userId,
            },
            include: {
                history: {
                    where: whereClause,
                    orderBy: {
                        createdAt: loyaltyPointHistoryDto.orderBy as 'asc' | 'desc',
                    },
                    skip: (loyaltyPointHistoryDto.page - 1) * loyaltyPointHistoryDto.numPerPage,
                    take: loyaltyPointHistoryDto.numPerPage,
                },
            },
        });

        // Get the total count of credit history records
        const totalCount = await prisma.loyaltyPointHistory.count({
            where: {
                loyaltyPointId: loyaltyPointHistory.id,
                ...whereClause,
            },
        });

        return {
            loyaltyPointHistory,
            totalCount,
        };
    }

    async getMembershipDetails(userId: string) {

        const membership = await prisma.loyaltyPoint.findUnique({
            where: { userId: userId }
        });

        if (!membership) {

            throw new NotFoundException({
                error: 'User not found. Please try again later.',
                message: ['User not found. Please try again later.'],
                statusCode: 500
            })
        }

        return membership;
    }

    async create(user: User) {

        const newMembership = await prisma.loyaltyPoint.create({
            data: {
                balance: 0.00,
                userId: user.id,
                createdAt: new Date(),
                user: { connect: { id: user.id } },
            },
        })

        // Update the user with the new creditId
        await prisma.user.update({
            where: { id: user.id },
            data: { loyaltyPointId: newMembership.id },
        });

        // Fetch the updated user
        return await this.getLoyaltyPointBalance(user.id)
    }

    async getLoyaltyPointBalance(userId: string) {

        const user = await prisma.user.findFirst({
            where: { id: userId },
            include: {
                loyaltyPoint: {
                    include: {
                        history: {
                            orderBy: {
                                createdAt: 'desc',
                            },
                            take: 5,
                        }
                    }
                },
                membershipTier: true
            }
        })

        // Remove fields you don't want to include in the response
        const { id, email, phoneNumber, dateOfBirth, addressId, profilePic, roleId, creditId, loyaltyPointId, ...newStructure } = user; // Adjust the fields to exclude as needed

        return newStructure;
    }

    async addLoyaltyPoints(userId: string, points: number, type: string = TransactionType.PURCHASE, orderId?: string): Promise<void> {
        const user = await prisma.user.findUnique({
            where: { id: userId },
            include: { loyaltyPoint: true },
        });

        if (!user || !user.loyaltyPoint) {

            throw new InternalServerErrorException({
                error: 'User or loyalty points not found. Please try again later.',
                message: ['User or loyalty points not found. Please try again later.'],
                statusCode: 500
            })
        }

        const newBalance = user.loyaltyPoint.balance + points;

        // Update loyalty points balance
        await prisma.loyaltyPoint.update({
            where: { userId: userId },
            data: { balance: newBalance },
        });

        // Record the history
        await prisma.loyaltyPointHistory.create({
            data: {
                userId: userId,
                amount: points,
                newTotal: newBalance,
                lastLoyaltyPointAmount: user.loyaltyPoint.balance,
                type: type,
                orderId: orderId,
                loyaltyPointId: user.loyaltyPoint.id,
                createdAt: new Date(),
                updatedAt: new Date()
            },
        });
    }

    /********** MerchantDize **********/

    async createProduct(file: Express.Multer.File, data: { name: string; category: string; description: string; price: number; pointsCost: number; images: string[]; stock: number; tagId?: string }) {

        try {
            const res = await prisma.merchandiseProduct.create({
                data,
            });

            const publicUrl = await UploadFunction.uploadimage("loyalty_merchant_img", res.id, file);
            const updatedImages = [publicUrl.data.publicUrl]

            return await prisma.merchandiseProduct.update({
                where: {
                    id: res.id
                },
                data: {
                    images: updatedImages
                }
            })
        } catch (error) {

            throw (error)
        }

    }

    async updateProduct(id: string, file: Express.Multer.File, data: { name?: string; description?: string; price?: number; pointsCost?: number; images?: string[]; stock?: number; tagId?: string }) {

        try {

            if (!isEmpty(file)) {
                const publicUrl = await UploadFunction.uploadimage("loyalty_merchant_img", id, file);
                let updatedImages = [publicUrl.data.publicUrl]
                data.images = updatedImages
            }

            return await prisma.merchandiseProduct.update({
                where: { id },
                data,
            });
        } catch (error) {

            throw (error)
        }
    }

    async deleteProduct(id: string) {

        try {

            return await prisma.merchandiseProduct.delete({
                where: { id },
            });
        } catch (error) {

            throw (error)
        }

    }

    async listProducts(listProductsDto: ListProductsDto) {

        const page = listProductsDto.page
        const numPerPage = listProductsDto.numPerPage
        const skip = (page - 1) * numPerPage;

        const [products, total] = await Promise.all([
            prisma.merchandiseProduct.findMany({
                skip,
                take: numPerPage,
                include: {
                    tag: true,
                },
                orderBy: {
                    createdAt: 'desc',
                },
            }),
            prisma.merchandiseProduct.count(),
        ]);

        return {
            total,
            page,
            numPerPage,
            data: products.map(product => ({
                ...product,
                category: product.category,
            })),
        };
    }

    async redeemProduct(redeemProduct: RedeemProductDto) {

        if (!redeemProduct.paymentMethod) {
            redeemProduct.paymentMethod = TransactionType.LOYALTY_POINT;
        }

        const user = await prisma.user.findUnique({
            where: { id: redeemProduct.userId },
            include: {
                loyaltyPoint: true,
                credit: true,
            },
        });

        const product = await prisma.merchandiseProduct.findUnique({
            where: { id: redeemProduct.productId },
        });

        if (!user || !product) {
            throw new InternalServerErrorException({
                error: 'Product or User not found. Please try again later.',
                message: ['Product or User not found. Please try again later.'],
                statusCode: 500
            })
        }

        if (product.stock <= 0) {
            throw new InternalServerErrorException({
                error: 'Product out of stock. Please try again later.',
                message: ['Product out of stock. Please try again later.'],
                statusCode: 500
            })
        }

        if (redeemProduct.paymentMethod === TransactionType.LOYALTY_POINT) {

            if (user.loyaltyPoint.balance < product.pointsCost) {

                throw new InternalServerErrorException({
                    error: 'Insufficient points',
                    message: ['You do not have enough points to redeem this gift.'],
                    statusCode: 500
                })
            }

            const updatedLP = await prisma.loyaltyPoint.update({
                where: { userId: redeemProduct.userId },
                data: {
                    balance: {
                        decrement: product.pointsCost,
                    },
                },
            });

            // Record the history
            await prisma.loyaltyPointHistory.create({
                data: {
                    userId: redeemProduct.userId,
                    amount: -product.pointsCost,
                    newTotal: updatedLP.balance,
                    lastLoyaltyPointAmount: user.loyaltyPoint.balance,
                    type: TransactionType.REDEMPTION,
                    loyaltyPointId: user.loyaltyPoint.id,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                },
            });
        }
        /*** Disable for now

        else if (paymentMethod === 'credit') {
            if (user.credit.balance < product.price) {
                throw new Error('Insufficient credit');
            }

            const walletData = await prisma.credit.update({
                where: { userId: userId },
                data: {
                    balance: {
                        decrement: product.price,
                    },
                },
            });

            // Create new CreditHistory entry
            await prisma.creditHistory.create({
                data: {
                    amount: product.price,
                    actualAmount: product.price,
                    type: PaymentType.PAID,
                    // orderId: orderId,
                    creditId: walletData.id,
                    lastCreditAmount: walletData.balance,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                },
            });
        }

        ***/

        else {
            throw new InternalServerErrorException({
                error: 'Invalid Payment Method. Please try again later.',
                message: ['Invalid Payment Method. Please try again later.'],
                statusCode: 500
            })
        }

        await prisma.merchandiseProduct.update({
            where: { id: redeemProduct.productId },
            data: {
                stock: {
                    decrement: 1,
                },
            },
        });

        return prisma.userRedemptionHistory.create({
            data: {
                userId: redeemProduct.userId,
                productId: redeemProduct.productId,
                pointsRedeemed: redeemProduct.paymentMethod === TransactionType.LOYALTY_POINT ? product.pointsCost : null,
                // creditRedeemed: paymentMethod === 'credit' ? product.price : null,
                deliveryMethod: redeemProduct.deliveryMethod,
                deliveryAddress: redeemProduct.deliveryAddress,
                merchantId: redeemProduct.deliveryMethod === DeliveryMethod.REDEEM_AT_STORE ? redeemProduct.merchantId : null,
            },
        });
    }

    async getUserRedemptionHistory(userRedemptionDto: UserRedemptionDto) {

        const whereClause: any = { userId: userRedemptionDto.userId };

        if (userRedemptionDto.startDate) {
            whereClause.createdAt = {
                ...whereClause.createdAt,
                gte: new Date(userRedemptionDto.startDate),
            };
        }

        if (userRedemptionDto.endDate) {
            whereClause.createdAt = {
                ...whereClause.createdAt,
                lte: new Date(userRedemptionDto.endDate),
            };
        }

        const res = await prisma.userRedemptionHistory.findMany({
            where: whereClause,
            orderBy: {
                createdAt: userRedemptionDto.orderBy as 'asc' | 'desc',
            },
            skip: (userRedemptionDto.page - 1) * userRedemptionDto.numPerPage,
            take: userRedemptionDto.numPerPage,
            include: {
                product: true,  // Include related product data
                merchant: true, // Include related merchant data
                user: true,
            },
        });

        // Hide UUIDs by mapping over the results and omitting the `id` fields
        const sanitizedRes = res.map(item => {
            return {
                ...item,
                // id: undefined,
                // userId: undefined,
                productId: undefined,
                // merchantId: undefined,
                user: {
                    ...item.user,
                    id: undefined,
                },
                // product: {
                //     ...item.product,
                //     id: undefined,
                // },
                // merchant: item.merchant ? {
                //     ...item.merchant,
                //     id: undefined,
                // } : null,
            };
        });

        // Get the total count of redemption history records
        const totalCount = await prisma.userRedemptionHistory.count({
            where: whereClause,
        });

        return {
            totalCount,
            redemptionHistory: sanitizedRes,
        };
    }
}
