// src/menu/menu.controller.ts

import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Param,
  Post,
  Put,
  UseInterceptors,
  UploadedFile
} from '@nestjs/common';
import { ApiTags } from '@nestjs/swagger';
import { FileInterceptor } from '@nestjs/platform-express';
import {
  CheckMenuAvailabilityDto,
  CreateMenuDto,
  UpdateMenuDto,
  MenuImgUploadDto
} from './menu.dto';
import { MenuService } from './menu.service';

@ApiTags('Menu')
@Controller('menu')
export class MenuController {
  constructor(private menuService: MenuService) {}
  
  @Get()
  @HttpCode(HttpStatus.OK)
  async findAll() {
    const data = await this.menuService.findAll();
    return { data: data, message: 'Menu Successfully Retrieved All'}
  }

  @Get(':id')
  @HttpCode(HttpStatus.OK)
  async findOne(@Param('id') id: string) {
    const data = await this.menuService.findOne(id);
    return { data: data, message: 'Menu Successfully Retrieved One'}
  }
  
  @Get('merchant/:id')
  @HttpCode(HttpStatus.OK)
  findAllBasedOnMerchant(@Param('id') id: string) {
    const data = this.menuService.findAllByMerchant(id);
    return { data: data, message: 'Menu Successfully Updated'}
  }

  @Post()
  @UseInterceptors(FileInterceptor('image'))
  @HttpCode(HttpStatus.OK)
  async create(@UploadedFile() image: Express.Multer.File, @Body() reqData: CreateMenuDto) {
    const data = await this.menuService.create(image, reqData);
    return { data: data, message: 'Menu Successfully Created'}
  }

  @Post('upload')
  @UseInterceptors(FileInterceptor('imagefile'))
  @HttpCode(HttpStatus.OK)
  async uploadMenuImg(@UploadedFile() imagefile: Express.Multer.File, @Body() reqData: MenuImgUploadDto){
    const data = await this.menuService.uploadMenuImage(imagefile, reqData);
    return {data: data, message: 'Menu Image Upload Successful'}
  }

  @Post('availability')
  @HttpCode(HttpStatus.OK)
  async checkMenuAvailability(@Body() reqData: CheckMenuAvailabilityDto) {
    const data = await this.menuService.findMenu(reqData);
    return { data: data, message: 'Menu Availability is Retrieved'}
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  async update(@Param('id') id: string, @Body() reqData: UpdateMenuDto) {
    const data = await this.menuService.update(id, reqData);
    return { data: data, message: 'Menu Successfully Updated'}
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  async remove(@Param('id') id: string) {
    const data = await this.menuService.remove(id);
    return { data: data, message: 'Menu Successfully Updated'}
  }
}
