// src/menu/menu.service.ts

import { Injectable } from '@nestjs/common';

import prisma from '../prisma';
import {
  CheckMenuAvailabilityDto,
  CreateMenuDto,
  UpdateMenuDto,
  MenuImgUploadDto,
  MenuOptionDto
} from './menu.dto';
import {
  UploadImg
} from 'src/utils';

const UploadFunction = new UploadImg();

@Injectable()
export class MenuService {
  constructor() {}

  async create(file: Express.Multer.File, data: CreateMenuDto) {
    const menuoption = JSON.parse(data.menuOption)

    const menu = await prisma.menu.create({
      data: {
        name: data.name,
        basePrice: parseFloat(data.basePrice),
        category: data.category,
        description: data.description,
        merchantId: data.merchantId,
        active: data.active,
        image: "",
        menuOption: {
          create: menuoption.map((menuOption) => ({
            name: menuOption.name,
            isRequired: menuOption.isRequired,
            min: menuOption.min,
            max: menuOption.max,
            active: menuOption.active,
            option: {
              create: menuOption.option.map((opt) => ({
                name: opt.name,
                price: opt.price,
                active: opt.active,
              })),
            },
          })),
        },
      },
    });

    const publicurl = await UploadFunction.uploadimage("menuimg", menu.id, file)
    // Update the menu item with the public URL of the image
    const updatedMenu = await prisma.menu.update({
      where: { id: menu.id },
      data: {
        image: publicurl.data.publicUrl,
      },
      include: {
        menuOption: {
          include: {
            option: true, // Include all options for each menu option
          },
        },
      },
    });

    return updatedMenu;
  }

  async findAll() {
    return await prisma.menu.findMany({
      include: {
        menuOption: {
          include: {
            option: true
          }
        }
      }
    });
  }

  async findAllByMerchant(merchantId: string) {
    return prisma.menu.findMany({
      where: { merchantId },
      include: {
        menuOption: {
          include:  {
            option: true
          }
        }
      },
    });
  }

  async findOne(id: string) {
    return prisma.menu.findUnique({ 
      where: { id },
      include: {
        menuOption: {
          include:  {
            option: true
          }
        }
      },
    });
  }

  async findMenu(data: CheckMenuAvailabilityDto) {
    return prisma.menu.findMany({
      where : {
        id : {
          in: data.menuIds
        }
      },
      include: {
        menuOption: {
          include:  {
            option: true
          }
        }
      },
    })
  }

  async update(id: string, data: UpdateMenuDto) {
     // Update menu options
     for (const menuOption of data.menuOption) {

      var menuoptionid = "";
      if (menuOption.id) {
        // Update existing menu option
        await prisma.menuOption.update({
          where: { id: menuOption.id },
          data: {
            name: menuOption.name,
            isRequired: menuOption.isRequired,
            min: menuOption.min,
            max: menuOption.max,
            active: menuOption.active
          },
        });
        menuoptionid = menuOption.id
      } else {
        // Create new menu option
        const menuoption = await prisma.menuOption.create({
          data: {
            name: menuOption.name,
            isRequired: menuOption.isRequired,
            min: menuOption.min,
            max: menuOption.max,
            ...(menuOption.active && {
              active: menuOption.active
            }),  
            menu:  {
              connect: {
                id: id,
              },
            },
          },
        });

        menuoptionid = menuoption.id
      }

      // Create associated options
      for (const option of menuOption.option) {
        if (option.id) {
          // Update existing option
          await prisma.option.update({
            where: { id: option.id },
            data: {
              name: option.name,
              price: option.price,
              active: option.active
            },
          });
        } else {
          // Create new option
          await prisma.option.create({
            data: {
              name: option.name,
              price: option.price,
              ...(option.active && {
                active: option.active
              }),  
              menuOption: {
                connect: {
                  id: menuoptionid,
                },
              },
            },
          });
        }
      }
    }

    const updatedMenu = await prisma.menu.update({
      where: { id },
      data: {
        name: data.name,
        basePrice: data.basePrice,
        description: data.description,
        active: data.active,
        category: data.category,
        merchantId: data.merchantId,
      },
      include: {
        menuOption: {
          include: {
            option: true, // Include associated options for each menu option
          },
        },
      },
    });

    return updatedMenu;
  }

  async remove(id: string) {
    return prisma.menu.delete({ 
      where: { id },
      include: {
        menuOption: {
          include: {
            option: true
          }
        }
      },
    });
  }

  async uploadMenuImage(file: Express.Multer.File, data: MenuImgUploadDto){
    const publicurl = await UploadFunction.uploadimage("menuimg", data.menuId, file)

    await prisma.menu.update({
      where: {  id: data.menuId }, 
      data:{
        image: publicurl.data.publicUrl
      } 
    })
    
    return publicurl.data.publicUrl
  }
}
