// src/merchant/merchant.controller.ts

import {
  calculateDistanceBetweenMerchantandUser,
  calculateMerchantRating,
} from 'src/utils';

import {
  Body,
  Controller,
  Delete,
  Get,
  HttpCode,
  HttpStatus,
  Logger,
  Param,
  Post,
  Put,
  UseInterceptors,
  UploadedFile
} from '@nestjs/common';
import { ApiTags } from '@nestjs/swagger';
import { FileInterceptor } from '@nestjs/platform-express';

import {
  CreateUpdateMerchantDto,
  GetMerchantDTO,
  MerchantImgUploadDto,
  NearbyShopDTO,
  ReviewMerchantDTO,
  UpdateOpeningHours,
  MerchantUpdateAddressDto
} from './merchant.dto';
// import { Merchant } from '@prisma/client';
import { MerchantService } from './merchant.service';

@ApiTags('Merchants')
@Controller('merchants')
export class MerchantController {
  constructor(private readonly merchantService: MerchantService) {}

  @Get('getallmerchant')
  @HttpCode(HttpStatus.OK)
  async findAll() {
    let data = await this.merchantService.findAll();

    data.forEach((elem, index) => {
      data[index]['overallRating'] = calculateMerchantRating(elem.rating);
    })

    return { data: data, message: 'Merchant Retrieved All'}
  }

  @Get('getallactivemerchant')
  @HttpCode(HttpStatus.OK)
  async findAllActiveMerchant() {
    let data = await this.merchantService.findAllActiveMerchant();

    data.forEach((elem, index) => {
      data[index]['overallRating'] = calculateMerchantRating(elem.rating);
    })

    return { data: data, message: 'Merchant Retrieved All'}
  }

  @Post('getmerchant')
  @HttpCode(HttpStatus.OK)
  async findOne(@Body() reqData: GetMerchantDTO) {
    const { id, latitude, longitude } = reqData;

    const userLatitude = parseFloat(latitude.toString());
    const userLongitude = parseFloat(longitude.toString());

    let data = await this.merchantService.findOne(id);
    data['overallRating'] = calculateMerchantRating(data.rating);
    data['distance'] = calculateDistanceBetweenMerchantandUser(userLatitude, userLongitude, data.address.latitude, data.address.longitude)

    return { data: data, message: 'Merchant Retrieved One'}
  }

  @Post('createmerchant')
  @HttpCode(HttpStatus.OK)
  async create(@Body() reqData: CreateUpdateMerchantDto) {
    const data = await this.merchantService.create(reqData);
    return { data: data, message: 'Merchant Created'}
  }

  @Post('upload')
  @UseInterceptors(FileInterceptor('imagefile'))
  @HttpCode(HttpStatus.OK)
  async uploadMenuImg(@UploadedFile() imagefile: Express.Multer.File, @Body() reqData: MerchantImgUploadDto){
    const data = await this.merchantService.uploadMenuImage(imagefile, reqData);
    return {data: data, message: 'Merchant Image Upload Successful'}
  }

  @Post('nearbyshop')
  @HttpCode(HttpStatus.OK)
  async nearbyshop(@Body() reqData: NearbyShopDTO) {

    const { latitude, longitude } = reqData;

    Logger.log(latitude, "nearbyshop")
    Logger.log(longitude, "nearbyshop")

    const data = await this.merchantService.findNearbyShop(latitude, longitude);
    return { data: data, message: 'Nearby Shop Data'}
  }

  @Put(':id/review')
  @HttpCode(HttpStatus.OK)
  async updateReview(@Param('id') id: string, @Body() reqData: ReviewMerchantDTO) {
    const data = await this.merchantService.updateReview(id, reqData);
    return { data: data, message: 'Merchant Review Updated'}
  }

  @Put(':id/openinghours')
  @HttpCode(HttpStatus.OK)
  async updateOpeningHours(@Param('id') id: string, @Body() reqData: UpdateOpeningHours) {
    const data = await this.merchantService.updateOpeningHours(id, reqData);
    return { data: data, message: 'Merchant Updated'}
  }

  @Put(':id/address')
  @HttpCode(HttpStatus.OK)
  async updateAddress(@Param('id') id: string, @Body() reqData: MerchantUpdateAddressDto) {
    const data = await this.merchantService.updateAddress(id, reqData);
    return { data: data, message: 'Merchant Updated'}
  }

  @Put(':id')
  @HttpCode(HttpStatus.OK)
  async update(@Param('id') id: string, @Body() reqData: CreateUpdateMerchantDto) {
    const data = await this.merchantService.update(id, reqData);
    return { data: data, message: 'Merchant Updated'}
  }

  @Delete(':id')
  @HttpCode(HttpStatus.OK)
  async remove(@Param('id') id: string) {
    const data = await this.merchantService.remove(id);
    return { data: data, message: 'Merchant Deleted'}
  }
}
