import { BadRequestException, Injectable } from '@nestjs/common';

import prisma from '../prisma';
import {
  CreateUpdateMerchantDto,
  ReviewMerchantDTO,
  MerchantImgUploadDto,
  UpdateOpeningHours,
  MerchantUpdateAddressDto
} from './merchant.dto';
import {
  isEmpty,
  UploadImg
} from 'src/utils';

@Injectable()
export class MerchantService {
  constructor() { }

  async findAll() {
    const allmerchant = await prisma.merchant.findMany({
      include: {
        address: true,
        menu: true,
        rating: {
          include: {
            user: true
          }
        },
        user: true
      }
    });

    const restructuredOrders = allmerchant.map((merchant) => ({
      ...merchant,
      openingHours: JSON.parse(merchant.openingHours)
    }));

    return restructuredOrders
  }

  async findAllActiveMerchant() {
    const allactive = await prisma.merchant.findMany({
      where: { active: true },
      include: {
        address: true,
        menu: true,
        rating: {
          include: {
            user: true
          }
        },
        user: true
      }
    });

    const restructuredOrders = allactive.map((merchant) => ({
      ...merchant,
      openingHours: JSON.parse(merchant.openingHours)
    }));

    return restructuredOrders
  }

  async findOne(id: string) {
    return prisma.merchant.findUnique({
      where: { id },
      include: {
        address: true,
        menu: true,
        rating: {
          include: {
            user: true
          }
        },
        user: true
      }
    });
  }

  async create(data: CreateUpdateMerchantDto) {
    const openinghours = data.openingHours ? data.openingHours : JSON.stringify({
      "sunday": { "open": "", "close": "", "rest": true },
      "monday": { "open": "", "close": "", "rest": true },
      "tuesday": { "open": "", "close": "", "rest": true },
      "wednesday": { "open": "", "close": "", "rest": true },
      "thursday": { "open": "", "close": "", "rest": true },
      "friday": { "open": "", "close": "", "rest": true },
      "saturday": { "open": "", "close": "", "rest": true }
    })

    const user = await prisma.user.findUnique({
      where: {
        email: data.email
      }
    })

    if (isEmpty(user)) {

      throw new BadRequestException({
        error: 'User email is not MaCoffee user yet. Please register as MaCoffee user first to proceed.',
        message: ['User email is not MaCoffee user yet. Please register as MaCoffee user first to proceed.'],
        statusCode: 400
      });
    }

    return prisma.merchant.create({
      data: {
        name: data.name,
        user: {
          connect: {
            id: user.id
          }
        },
        storyDescription: data.name,
        openingHours: openinghours,
        businessType: data.businessType,
        ...(data.image && {
          image: data.image
        }),
        ...(data.businessRegistrationNumber && {
          businessRegistrationNumber: data.businessRegistrationNumber
        }),
        ...(data.contractSigneeName && {
          contractSigneeName: data.contractSigneeName
        }),
        ...(data.contractSigneeNRIC && {
          contractSigneeNRIC: data.contractSigneeNRIC
        }),
        address: {
          create: {
            address: data.address,
            city: data.city,
            state: data.state,
            country: data.country,
            postalCode: data.postalCode
          }
        }
        // ...(data.userId && { userId: data.userId }),
      },
    });
  }

  async update(id: string, data: CreateUpdateMerchantDto) {
    const merchant = await prisma.merchant.findUnique({
      where: { id },
      include: {
        address: true
      }
    });

    return prisma.merchant.update({
      where: { id },
      data: {
        name: data.name,
        storyDescription: data.storyDescription,
        businessType: data.businessType,
        ...(data.image && {
          image: data.image
        }),
        ...(data.businessRegistrationNumber && {
          businessRegistrationNumber: data.businessRegistrationNumber
        }),
        ...(data.contractSigneeName && {
          contractSigneeName: data.contractSigneeName
        }),
        ...(data.contractSigneeNRIC && {
          contractSigneeNRIC: data.contractSigneeNRIC
        }),
        address: {
          update: {
            where: { id: merchant.addressId },
            data: {
              address: data.address,
              city: data.city,
              state: data.state,
              country: data.country,
              postalCode: data.postalCode,
              latitude: data.latitude,
              longitude: data.longitude
            },
          },
        },
      }
    });
  }

  async updateOpeningHours(id: string, data: UpdateOpeningHours) {
    return prisma.merchant.update({
      where: { id },
      data: {
        openingHours: JSON.stringify(data.openingHours)
      }
    })
  }

  async updateAddress(id: string, data: MerchantUpdateAddressDto) {
    return prisma.address.update({
      where: { id },
      data: {
        ...data,
        updatedAt: new Date(),
      }
    })
  }

  async updateReview(id: string, data: ReviewMerchantDTO) {
    return prisma.merchant.update({
      where: { id },
      data: {
        rating: {
          create: {
            reviewerId: data.reviewerId,
            orderId: data.orderId,
            rating: data.rating,
            review: data.review
          }
        }
      }
    })
  }

  async remove(id: string) {
    return prisma.merchant.delete({
      where: { id },
    });
  }

  async findNearbyShop(latitude: number, longitude: number) {
    var shops = await prisma.merchant.findMany({
      where: { active: true },
      include: {
        address: true
      }
    });

    if (!latitude || !longitude) {
      return { error: 'Latitude and longitude are required.' };
    }

    const userLatitude = parseFloat(latitude.toString());
    const userLongitude = parseFloat(longitude.toString());

    if (isNaN(userLatitude) || isNaN(userLongitude)) {
      return { error: 'Invalid latitude or longitude.' };
    }

    const nearbyShops = await this.filterNearbyShops(userLatitude, userLongitude, shops);

    return nearbyShops;
  }

  async filterNearbyShops(userLatitude, userLongitude, shops) {
    const nearbyShops = [];

    shops.forEach(async shop => {
      const distance = await this.calculateDistance(
        userLatitude,
        userLongitude,
        shop.address.latitude,
        shop.address.longitude
      );

      if (distance <= 10) {
        nearbyShops.push({ ...shop, distance });
      }
    });

    return nearbyShops;
  }

  async calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371; // Earth's radius in kilometers
    const dLat = (lat2 - lat1) * (Math.PI / 180);
    const dLon = (lon2 - lon1) * (Math.PI / 180);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos((lat1) * (Math.PI / 180)) * Math.cos((lat2) * (Math.PI / 180)) *
      Math.sin(dLon / 2) * Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    const distance = R * c; // Distance in kilometers
    return distance;
  }

  async uploadMenuImage(file: Express.Multer.File, data: MerchantImgUploadDto) {
    const UploadFunction = new UploadImg();
    const publicurl = await UploadFunction.uploadimage("merchantprofilepic", data.merchantId, file)

    await prisma.merchant.update({
      where: { id: data.merchantId },
      data: {
        image: publicurl.data.publicUrl
      }
    })

    return publicurl.data.publicUrl
  }
}
