import { Injectable } from '@nestjs/common';

import prisma from '../prisma';
import {
  CreateUpdateOrderDto,
  UpdateOrderStatusdto,
  userOrderHistoryDto,
  merchantOrderHistoryDto
} from './order.dto';
import { MembershipService } from 'src/membership/membership.service';
import { TransactionType } from 'src/membership/membership.dto';
import { generateReferenceId } from 'src/utils';

@Injectable()
export class OrderService {
  constructor(
    private readonly membershipService: MembershipService
  ) { }

  async create(data: CreateUpdateOrderDto) {
    let totalPrice = 0;

    const orderDetails = await Promise.all(
      data.orderDetails.map(async (orderDetail) => {
        const menu = await prisma.menu.findUnique({ where: { id: orderDetail.menuId } });
        const basePrice = menu.basePrice;

        const optionPrices = await Promise.all(
          orderDetail.orderDetailOptions.map(async (detailOption) => {
            const option = await prisma.option.findUnique({ where: { id: detailOption.optionId } });
            return option.price;
          })
        );

        const totalOptionPrice = optionPrices.reduce((acc, price) => acc + price, 0);
        const totalDetailPrice = (basePrice + totalOptionPrice) * orderDetail.quantity;

        totalPrice += totalDetailPrice;

        return {
          basePrice,
          quantity: orderDetail.quantity,
          menuId: orderDetail.menuId,
          orderDetailOptions: {
            create: await Promise.all(
              orderDetail.orderDetailOptions.map(async (detailOption) => {
                const menuOption = await prisma.menuOption.findUnique({
                  where: { id: detailOption.menuOptionId },
                });

                const option = await prisma.option.findUnique({
                  where: { id: detailOption.optionId },
                });

                return {
                  type: menuOption.name,
                  name: option.name,
                  price: option.price,
                };
              })
            ),
          },
        };
      })
    );

    let referenceId = data.referenceId;
    let attempt = 0;
    const maxAttempts = 10; // Set a reasonable limit for attempts

    if (!referenceId) {
      if (attempt >= maxAttempts) {
        throw new Error('Unable to generate a unique reference ID after multiple attempts');
      }

      let exists: boolean;
      do {
        referenceId = generateReferenceId();
        const res = await prisma.order.findUnique({
          where: { referenceId },
        });

        exists = !!res; // Set exists to true if a record is found, otherwise false
         attempt++;
      } while (exists);
    }

    return prisma.order.create({
      data: {
        referenceId: referenceId,
        totalAmount: totalPrice,
        orderTime: new Date(data.orderTime),
        pickupTime: new Date(data.pickupTime),
        userId: data.userId,
        status: "pending",
        ...(data.transactionId && {
          transactionId: data.transactionId,
        }),
        ...(data.vehicleId && {
          userVehicleId: data.vehicleId,
        }),
        ...(data.merchantId && {
          merchantId: data.merchantId,
        }),
        orderDetails: {
          create: orderDetails,
        },
      },
      include: {
        orderDetails: {
          include: {
            orderDetailOptions: true,
          },
        },
      },
    });
  }

  async findAll() {
    return prisma.order.findMany({
      include: {
        orderDetails: {
          include: {
            orderDetailOptions: true
          }
        },
        userVehicle: true,
        transaction: true,
        rating: true,
        user: true
      }
    });
  }

  async findOne(id: string) {
    const order = await prisma.order.findUnique({
      where: { id },
      include: {
        orderDetails: {
          include: {
            orderDetailOptions: true,
            menu: {
              select: {
                name: true,
                image: true,
              }
            },
          }
        },
        merchant: {
          select: {
            id: true,
            name: true,
            address: true
          }
        },
        userVehicle: true,
        transaction: true,
        rating: true,
        user: true
      }
    });

    const restructuredOrder = {
      id: order.id,
      voucherUsed: order.voucherUsed,
      totalAmount: order.totalAmount,
      orderTime: order.orderTime,
      pickupTime: order.pickupTime,
      status: order.status,
      createdAt: order.createdAt,
      updatedAt: order.updatedAt,
      transactionId: order.transactionId,
      merchantId: order.merchant.id,
      merchantAddress: order.merchant.address,
      merchantName: order.merchant.name,
      orderDetails: order.orderDetails.map(orderDetail => ({
        id: orderDetail.id,
        quantity: orderDetail.quantity,
        basePrice: orderDetail.basePrice,
        createdAt: orderDetail.createdAt,
        updatedAt: orderDetail.updatedAt,
        orderId: orderDetail.orderId,
        menuId: orderDetail.menuId,
        menuName: orderDetail.menu?.name || null, // Null check in case menu is not present
        menuImage: orderDetail.menu?.image || null,
        orderDetailOptions: orderDetail.orderDetailOptions,
      })),
      transaction: order.transaction,
      rating: order.rating,
      vehicle: order.userVehicle,
      user: order.user
    };

    return restructuredOrder
  }

  async findByUserId(data: userOrderHistoryDto) {

    const pageSize = data.numPerPage;
    const skip = (data.page - 1) * pageSize;

    const orders = await prisma.order.findMany({
      take: pageSize,
      skip: skip,
      where: { userId: data.userId },
      include: {
        orderDetails: {
          include: {
            orderDetailOptions: true,
            menu: {
              select: {
                name: true,
                image: true,
              }
            },
          }
        },
        merchant: {
          select: {
            id: true,
            name: true,
            address: true
          }
        },
        userVehicle: true,
        transaction: true,
        rating: true,
        user: true
      }
    });

    const restructuredOrders = orders.map(order => ({
      id: order.id,
      totalAmount: order.totalAmount,
      orderTime: order.orderTime,
      pickupTime: order.pickupTime,
      status: order.status,
      createdAt: order.createdAt,
      updatedAt: order.updatedAt,
      transactionId: order.transactionId,
      merchantId: order.merchant.id,
      merchantAddress: order.merchant.address,
      merchantName: order.merchant.name,
      orderDetails: order.orderDetails.map(orderDetail => ({
        id: orderDetail.id,
        quantity: orderDetail.quantity,
        basePrice: orderDetail.basePrice,
        createdAt: orderDetail.createdAt,
        updatedAt: orderDetail.updatedAt,
        orderId: orderDetail.orderId,
        menuId: orderDetail.menuId,
        menuName: orderDetail.menu?.name || null, // Null check in case menu is not present
        menuImage: orderDetail.menu?.image || null,
        orderDetailOptions: orderDetail.orderDetailOptions,
      })),
      transaction: order.transaction,
      rating: order.rating,
      vehicle: order.userVehicle,
      user: order.user
    }));

    return restructuredOrders
  }

  async findByMerchantId(data: merchantOrderHistoryDto) {
    const pageSize = data.numPerPage;
    const skip = (data.page - 1) * pageSize;

    const [orders, totalCount] = await Promise.all([
      prisma.order.findMany({
        take: pageSize,
        skip: skip,
        where: {
          merchantId: data.merchantId,
          ...(data.status && {
            status: data.status,
          }),
          ...(data.orderTime && {
            orderTime: data.orderTime,
          }),
          ...(data.pickupTime && {
            pickupTime: data.pickupTime,
          }),
          ...(data.userId && {
            userId: data.userId,
          }),
        },
        include: {
          orderDetails: {
            include: {
              orderDetailOptions: true,
              menu: {
                select: {
                  name: true,
                  image: true,
                },
              },
            },
          },
          merchant: {
            select: {
              id: true,
              name: true,
              address: true,
            },
          },
          userVehicle: true,
          transaction: true,
          rating: true,
          user: true,
        },
      }),
      prisma.order.count({
        where: {
          merchantId: data.merchantId,
          ...(data.status && {
            status: data.status,
          }),
          ...(data.orderTime && {
            orderTime: data.orderTime,
          }),
          ...(data.pickupTime && {
            pickupTime: data.pickupTime,
          }),
          ...(data.userId && {
            userId: data.userId,
          }),
        },
      }),
    ]);

    const restructuredOrders = orders.map((order) => ({
      id: order.id,
      totalAmount: order.totalAmount,
      orderTime: order.orderTime,
      pickupTime: order.pickupTime,
      status: order.status,
      createdAt: order.createdAt,
      updatedAt: order.updatedAt,
      transactionId: order.transactionId,
      merchantId: order.merchant.id,
      merchantAddress: order.merchant.address,
      merchantName: order.merchant.name,
      orderDetails: order.orderDetails.map((orderDetail) => ({
        id: orderDetail.id,
        quantity: orderDetail.quantity,
        basePrice: orderDetail.basePrice,
        createdAt: orderDetail.createdAt,
        updatedAt: orderDetail.updatedAt,
        orderId: orderDetail.orderId,
        menuId: orderDetail.menuId,
        menuName: orderDetail.menu?.name || null,
        menuImage: orderDetail.menu?.image || null,
        orderDetailOptions: orderDetail.orderDetailOptions,
      })),
      transaction: order.transaction,
      rating: order.rating,
      vehicle: order.userVehicle,
      user: order.user,
    }));

    return { orders: restructuredOrders, totalCount };
  }

  async update(id: string, data: CreateUpdateOrderDto) {
    return prisma.order.update({
      where: {
        id: id
      },
      data: {
        totalAmount: data.totalAmount,
        orderTime: new Date(),
        pickupTime: data.pickupTime,
        userId: data.userId,
        ...(data.transactionId && {
          transactionId: data.transactionId
        }),
        ...(data.vehicleId && {
          vehicleId: data.vehicleId,
        }),
        ...(data.merchantId && {
          merchantId: data.merchantId
        }),
        orderDetails: {
          createMany: {
            data: data.orderDetails.map((elem) => ({
              basePrice: 0,
              quantity: elem.quantity,
              menuId: elem.menuId,
              orderDetailOption: {
                create: elem.orderDetailOptions.map(option => ({
                  type: "",
                  name: "",
                  price: 0,
                })),
              },
            }))
          },
        }
      }
    });
  }

  private async updateMembershipTier(userId: string, newPoints: number): Promise<void> {
    const tiers = await prisma.tier.findMany({
      orderBy: { pointsRequired: 'asc' },
    });

    for (const tier of tiers) {
      if (newPoints >= tier.pointsRequired) {

        await prisma.user.update({
          where: { id: userId },
          data: { membershipTierId: tier.id },
        });
      }
    }
  }

  async updateStatus(id: string, data: UpdateOrderStatusdto) {

    const order = await this.findOne(id);
    let pointsToAdd = 0;

    if (!order) {
      throw new Error('Order not found');
    }

    // if not voucherUsed then give loyalty points
    if (!order?.voucherUsed && data.status.toLocaleLowerCase() === "completed") {
      const user = await prisma.user.findUnique({
        where: { id: order.user.id },
        include: {
          membershipTier: true,
          loyaltyPoint: true
        },
      });

      const tier = await prisma.tier.findUnique({
        where: { id: user.membershipTierId },
      });

      pointsToAdd = Math.floor(order.totalAmount * tier.pointsMultiplier);

      // update points and tiering
      const newAccumulatedPoints = user.loyaltyPoint.balance + pointsToAdd;
      await this.membershipService.addLoyaltyPoints(user.id, pointsToAdd, TransactionType.EARNED, order.id);
      await this.updateMembershipTier(user.id, newAccumulatedPoints);
    }

    return prisma.order.update({
      where: { id },
      data: {
        status: data.status,
        completedAt: data.status.trim().toLowerCase() === 'completed' ? new Date() : null,
      }
    });
  }

  async remove(id: string) {
    return prisma.order.delete({
      where: { id },
      include: {
        orderDetails: {
          include: {
            orderDetailOptions: true
          }
        },
        transaction: true
      }
    });
  }
}
