import { Body, Controller, Get, HttpCode, HttpStatus, Post, Query, Res } from '@nestjs/common';
import { ApiTags } from '@nestjs/swagger';
import { Response } from 'express';
import { CreditService } from 'src/credit/credit.service';
import { BillplzCallbackDto, PaywithCreditsDto, ReloadDto } from './payment.dto';
import { PaymentService } from './payment.service';

@ApiTags('payment')
@Controller('payment')
export class PaymentController {

    constructor(
        private paymentService: PaymentService,
        private creditService: CreditService
    ) { }

    @Post('reload')
    @HttpCode(HttpStatus.OK)
    async reload(@Body() reloadDto: ReloadDto) {

        const paymentData = await this.paymentService.createBillplzPaymentRequest(reloadDto.userId, reloadDto.amount)
        return { data: paymentData, message: 'Reload Amount for user: ' + reloadDto.userId }
    }

    @Post('credit')
    @HttpCode(HttpStatus.OK)
    async paymentWithCredits(@Body() paywithCreditsDto: PaywithCreditsDto) {

        const result = await this.paymentService.payWithCredits(paywithCreditsDto.userId,
            paywithCreditsDto.orderId, paywithCreditsDto.amount, paywithCreditsDto.paymentType)

        return { data: result, message: 'Payment with Credits for user: ' + paywithCreditsDto.userId }
    }

    @Post('billplz/callback')
    async handleBillplzCallback(@Body() callbackData: BillplzCallbackDto) {

        // Validate the request signature
        // const signature = callbackData.x_signature;
        // const expectedSignature = crypto
        //     .createHmac('sha256', process.env.BILLPLZ_SECRET_KEY)
        //     .update(JSON.stringify(callbackData))
        //     .digest('hex');

        // if (signature !== expectedSignature) {
        //     throw new BadRequestException('Invalid signature');
        // }

        console.log('call back triggered :: ', callbackData)
        if (callbackData.paid) {

            try {
                await this.creditService.addCredit(callbackData);
                return { status: 'Payment recorded' };
            } catch (error) {
                return { status: 'Error recording payment', error: error.message };
            }
        } else {
            return { status: 'Payment not completed' };
        }
    }

    @Get('result')
    async handleBillplzRedirectUrl(@Query() query: any, @Res() res: Response): Promise<any> {

        // Decode the query parameters if necessary
        const decodedParams = Object.keys(query).reduce((acc, key) => {
            const decodedKey = decodeURIComponent(key);
            acc[decodedKey] = query[key];
            return acc;
        }, {});

        if (decodedParams['billplz']['paid']?.trim() === "true") {

            res.redirect('/payment/transaction-success?id=' + decodedParams['billplz']['id'] + '&paid_at=' + decodedParams['billplz']['paid_at']);
        } else {

            res.redirect('/payment/transaction-failed?id=' + decodedParams['billplz']['id'] + '&paid_at=' + decodedParams['billplz']['paid_at']);
        }

        // // Process the decoded parameters as needed
        // return {
        //     message: 'Received parameters',
        //     params: decodedParams,
        // };
    }

    @Get('transaction-success')
    transactionSuccess(@Query() successRequestDto: any, @Res() res: Response): void {
        const { id, paid, paid_at } = successRequestDto;

        const receiptHtml = `
            <!DOCTYPE html>
            <html lang="en">
            <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Transaction Success</title>
                <style>
                body {
                    font-family: Arial, sans-serif;
                    background-color: #f5f5f5;
                    margin: 0;
                    padding: 20px;
                }
                .container {
                    max-width: 600px;
                    margin: 0 auto;
                    background-color: #fff;
                    padding: 20px;
                    border-radius: 8px;
                    box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
                }
                h1 {
                    color: #4CAF50;
                }
                .receipt {
                    margin-top: 20px;
                }
                .receipt-item {
                    margin-bottom: 10px;
                }
                .receipt-item span {
                    display: inline-block;
                    width: 150px;
                    font-weight: bold;
                }
                </style>
            </head>
            <body>
                <div class="container">
                <h1>Transaction Successful</h1>
                <p>Thank you for your payment. Your transaction was successful.</p>
                <div class="receipt">
                    <div class="receipt-item">
                    <span>Transaction ID:</span> ${id}
                    </div>

                    <div class="receipt-item">
                    <span>Date:</span> ${paid_at}
                    </div>
                </div>
                </div>
            </body>
            </html>
         `;

        res.send(receiptHtml);
    }

    @Get('transaction-failed')
    transactionFailed(@Query() query: any, @Res() res: Response): void {
        const { id, paid, paid_at } = query;

        const failureHtml = `
          <!DOCTYPE html>
          <html lang="en">
          <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>Transaction Failed</title>
            <style>
              body {
                font-family: Arial, sans-serif;
                background-color: #f5f5f5;
                margin: 0;
                padding: 20px;
              }
              .container {
                max-width: 600px;
                margin: 0 auto;
                background-color: #fff;
                padding: 20px;
                border-radius: 8px;
                box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);
              }
              h1 {
                color: #f44336;
              }
            </style>
          </head>
          <body>
            <div class="container">
              <h1>Transaction Failed</h1>
              <p>Unfortunately, your transaction could not be completed. Please try again.</p>
              <div class="receipt">
                    <div class="receipt-item">
                    <span>Transaction ID:</span> ${id}
                    </div>
                </div>
            </div>
          </body>
          </html>
        `;

        res.send(failureHtml);
    }
}
