import { HttpService } from '@nestjs/axios';
import { Inject, Injectable } from '@nestjs/common';
import { firstValueFrom } from 'rxjs';
import prisma from 'src/prisma';
import { BILLPLZ_CONFIG, BillplzConfig } from './config/BillplzConfig';
import { TransactionType } from 'src/membership/membership.dto';
import { PaymentType } from 'src/credit/credit.dto';

@Injectable()
export class PaymentService {

    private readonly apiKey: string;
    private readonly collectionId: string;
    private readonly billplzBaseUrl: string;

    constructor(
        @Inject(BILLPLZ_CONFIG) private config: BillplzConfig,
        private readonly httpService: HttpService
    ) {
        this.apiKey = config.apiKey;
        this.collectionId = config.collectionId;
        this.billplzBaseUrl = config.billplzBaseUrl;
    }

    async payWithCredits(userId: string, orderId: string, amount: number, paymentType: string = TransactionType.PURCHASE) {

        // check credit balance based on userId
        const walletData = await prisma.credit.findUnique({
            where: {
                userId
            }
        })

        // credit compare with amount 
        if (amount > walletData.balance) {

            return "Insufficient Funds"
        }

        // payment ==> record credit balance, update order
        const order = await prisma.order.findUnique({
            where: { id: orderId },
        })

        if (!order) {

            return "No Order data"
        }

        const transaction = await prisma.transaction.create({
            data: {
                totalAmount: amount,
                paymentTime: new Date(),
                paymentMethod: paymentType,
                createdAt: new Date(),
                orderId: order.id
            }
        })

        if (transaction) {

            // update back orders
            const order = prisma.order.update({
                where: { id: orderId },
                data: {
                    transactionId: transaction.id
                }
            })

            // Create new CreditHistory entry
            await prisma.creditHistory.create({
                data: {
                    amount: amount,
                    actualAmount: amount,
                    type: paymentType,
                    orderId: orderId,
                    creditId: walletData.id,
                    status: PaymentType.PAID,
                    lastCreditAmount: walletData.balance,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                },
            });

            // updateWallet
            const updatedCreditData = await prisma.credit.update({
                data: {
                    balance: (walletData.balance - amount)
                },
                where: {
                    userId
                }
            })
        }

        return "Payment recorded"

        // TODO rollback ?
    }

    async createBillplzPaymentRequest(userId: string, amount: number, orderId?: string) {

        try {

            const user = await prisma.user.findUnique({
                where: {
                    id: userId
                }
            });

            const response = await firstValueFrom(this.httpService.post(`${this.billplzBaseUrl}/bills`, {
                collection_id: this.collectionId,
                email: user.email, // Replace with actual customer email
                name: user.name, // Replace with actual customer name
                amount: amount * 100, // Billplz expects amount in cents
                callback_url: process.env.BILLPLZ_CALLBACK_URL,
                redirect_url: process.env.BILLPLZ_REDIRECT_URL,
                description: 'Top up credit',
            }, {
                auth: {
                    username: this.apiKey,
                    password: '', // Password can be empty for basic auth with API key
                },
            }));

            return response.data;

        } catch (error) {

            return { error: "Error During calling to Billplz API via Reload API :: " + error.response }
        }


    }

}
