import { Injectable } from '@nestjs/common';

import prisma from '../prisma';
import {
  CreateUpdateUserDto,
  UpdateVehicleDto,
  UserImgUploadDto
} from './user.dto';
import {
  UploadImg
} from 'src/utils';
import { Multer } from 'multer';


@Injectable()
export class UserService {
  constructor() {}

  async create(data: CreateUpdateUserDto) {
    const role = await prisma.role.findFirst({
      where : {
        name: data.role
      }
    });

    if(!role) {
      throw new Error('Role Customer Not Found');
    }
    
    return prisma.user.create({ 
      data:{
        email: data.email,
        name: data.name,
        phoneNumber: data.phoneNumber,
        dateOfBirth: new Date(data.dateOfBirth),
        role : {
          connect : {
            id: role.id
          }
        },
        address : {
          create : {
            address : "",
            city :"",
            state : "",
            country : "",
            postalCode : ""
          }
      }
    }});
  }

  async findAll() {
    return prisma.user.findMany({
      include : {
        address : true,
        merchant: true,
        role: true,
        userVehicle: true,
        reviews: true
      }
    });
  }

  async findOne(id: string) {
    return prisma.user.findUnique({ 
      where: { id },
      include : {
        address : true,
        merchant: true,
        role: true,
        userVehicle: true,
        reviews: true
      }
    });
  }

  async findOneByEmail(email: string) {
    return prisma.user.findUnique({ 
      where: { email },
      include : {
        address : true,
        merchant: true,
        role: true,
        userVehicle: true,
        reviews: true
      }
    });
  }

  async update(id: string, data: CreateUpdateUserDto) {
    const user = await prisma.user.findUnique({
      where : { id },
      include : {
        address : true
      }
    });

    return prisma.user.update({ 
      where: { id }, 
      data:{
        email: data.email,
        name: data.name,
        phoneNumber: data.phoneNumber,
        dateOfBirth: new Date(data.dateOfBirth)
    }});
  }

  async updateVehicle(id: string, data: UpdateVehicleDto) {
    if(data.id) {
      await prisma.userVehicle.update({
        where : { id : data.id },
        data : {
          plate : data.plate
        }
      })

      return await prisma.userVehicle.findUnique({
        where : { id : data.id }
      })
    } else {
      await prisma.user.update({
        where : { id },
        data : {
          userVehicle : {
            create : {
              plate : data.plate
            }
          }
        }
      })

      return await prisma.user.findUnique({
        where : { id },
        include : {
          userVehicle : true
        }
      })
    }
  }

  async remove(id: string) {
    return prisma.user.delete({ where: { id } });
  }

  async uploadMenuImage(file: Express.Multer.File, data: UserImgUploadDto){
    const UploadFunction = new UploadImg();
    const publicurl = await UploadFunction.uploadimage("userprofilepic", data.userId, file)

    await prisma.user.update({
      where: {  id: data.userId }, 
      data:{
        profilePic: publicurl.data.publicUrl
      } 
    })
    
    return publicurl.data.publicUrl
  }
}
