import { Injectable } from '@nestjs/common';
import { Rating } from '@prisma/client';
import {
  createClient,
  SupabaseClient,
} from '@supabase/supabase-js';
import { randomBytes } from 'crypto';
import * as fs from 'fs';
import { promisify } from 'util';

const readFileAsync = promisify(fs.readFile);

export function calculateMerchantRating(rating: Rating[]) {
  return rating.reduce((total, elem) => total + (elem.rating || 0), 0)
}

export function calculateDistanceBetweenMerchantandUser(lat1, lon1, lat2, lon2) {
  const R = 6371; // Earth's radius in kilometers
  const dLat = (lat2 - lat1) * (Math.PI / 180);
  const dLon = (lon2 - lon1) * (Math.PI / 180);
  const a =
    Math.sin(dLat / 2) * Math.sin(dLat / 2) +
    Math.cos((lat1) * (Math.PI / 180)) * Math.cos((lat2) * (Math.PI / 180)) *
    Math.sin(dLon / 2) * Math.sin(dLon / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  const distance = R * c; // Distance in kilometers
  return distance;
}

@Injectable()
export class UploadImg {
  private supabase: SupabaseClient;

  constructor() {
    this.supabase = createClient(
      process.env.SUPABASE_URL,
      process.env.ROW_LEVEL_SECURITY_KEY,
    );
  }

  async checkImageExists(bucketName: string, folderName: string, imageName: string): Promise<boolean> {
    // List objects in the bucket
    const { data: objects, error } = await this.supabase.storage.from(bucketName).list(folderName);
    if (error) {
      throw error;
    }

    // Check if the desired image name exists in the list of objects
    return objects.some(object => object.name === imageName);
  }

  async convertFileToBase64(file: Express.Multer.File) {
    const base64String = file.buffer.toString('base64');
    return { mimetype: file.mimetype, base64String: base64String };
  }

  async uploadimage(filename: string, imagename: string, file: Express.Multer.File) {

    const checkImageExists = await this.checkImageExists("image", filename, imagename);

    if (checkImageExists) {
      await this.supabase.storage.from('image').remove([`${filename}/${imagename}`])
    }

    const { mimetype, base64String } = await this.convertFileToBase64(file);
    const base64Data = base64String.replace(/^data:image\/\w+;base64,/, '');
    const buffer = Buffer.from(base64Data, 'base64');

    const { data, error } = await this.supabase.storage.from('image').upload(`${filename}/${imagename}`, buffer, {
      contentType: mimetype // Set MIME type in options
    });

    if (error) {
      throw error;
    }

    const key = data.path;
    const publicUrl = this.supabase.storage.from('image').getPublicUrl(`${filename}/${imagename}`);

    return publicUrl;
  }

}

export function isEmpty(value: any): boolean {
  return (
    value === null ||
    value === undefined ||
    (typeof value === 'string' && value.trim().length === 0) ||
    (Array.isArray(value) && value.length === 0) ||
    (typeof value === 'object' && Object.keys(value).length === 0)
  );
}

export function generateReferenceId() {

  const now = new Date();
  const year = now.getFullYear().toString().slice(-2); // Last two digits of the year
  const month = String(now.getMonth() + 1).padStart(2, '0'); // MM
  const day = String(now.getDate()).padStart(2, '0'); // DD
  const datePrefix = `${year}${month}${day}`; // YYMMDD

  // Generate a shorter random part (4 bytes = 8 hex characters)
  const randomSuffix = randomBytes(4).toString('hex'); // 4 bytes = 8 hex characters

  return `ORD-${datePrefix}-${randomSuffix}`;
}